"use strict";
/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SeedStrategy = exports.ReservedKeysVerificationStrategy = exports.EdgeLabelVerificationStrategy = exports.ReadOnlyStrategy = exports.LambdaRestrictionStrategy = exports.EarlyLimitStrategy = exports.GraphFilterStrategy = exports.RepeatUnrollStrategy = exports.CountStrategy = exports.PathRetractionStrategy = exports.PathProcessorStrategy = exports.OrderLimitStrategy = exports.MatchPredicateStrategy = exports.LazyBarrierStrategy = exports.InlineFilterStrategy = exports.IncidentToAdjacentStrategy = exports.IdentityRemovalStrategy = exports.FilterRankingStrategy = exports.AdjacentToIncidentStrategy = exports.MatchAlgorithmStrategy = exports.VertexProgramStrategy = exports.ProductiveByStrategy = exports.SubgraphStrategy = exports.PartitionStrategy = exports.OptionsStrategy = exports.HaltedTraverserStrategy = exports.ElementIdStrategy = exports.ConnectiveStrategy = exports.TraversalStrategy = exports.TraversalStrategies = void 0;
const traversal_js_1 = require("./traversal.cjs");
class TraversalStrategies {
    strategies;
    /**
     * Creates a new instance of TraversalStrategies.
     * @param {TraversalStrategies} [parent] The parent strategies from where to clone the values from.
     * @constructor
     */
    constructor(parent) {
        if (parent) {
            // Clone the strategies
            this.strategies = [...parent.strategies];
        }
        else {
            this.strategies = [];
        }
    }
    /** @param {TraversalStrategy} strategy */
    addStrategy(strategy) {
        this.strategies.push(strategy);
    }
    /** @param {TraversalStrategy} strategy */
    removeStrategy(strategy) {
        const idx = this.strategies.findIndex((s) => s.fqcn === strategy.fqcn);
        if (idx !== -1) {
            return this.strategies.splice(idx, 1)[0];
        }
        return undefined;
    }
    /**
     * @param {Traversal} traversal
     * @returns {Promise}
     */
    applyStrategies(traversal) {
        // Apply all strategies serially
        return this.strategies.reduce((promise, strategy) => promise.then(() => strategy.apply(traversal)), Promise.resolve());
    }
}
exports.TraversalStrategies = TraversalStrategies;
class TraversalStrategy {
    fqcn;
    configuration;
    connection;
    /**
     * @param {String} fqcn fully qualified class name in Java of the strategy
     * @param {TraversalStrategyConfiguration} configuration for the strategy
     */
    constructor(fqcn, configuration = {}) {
        this.fqcn = fqcn;
        this.configuration = configuration;
    }
    /**
     * @abstract
     * @param {Traversal} traversal
     * @returns {Promise}
     */
    async apply(traversal) { }
}
exports.TraversalStrategy = TraversalStrategy;
class ConnectiveStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.decoration.ConnectiveStrategy');
    }
}
exports.ConnectiveStrategy = ConnectiveStrategy;
class ElementIdStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.decoration.ElementIdStrategy');
    }
}
exports.ElementIdStrategy = ElementIdStrategy;
class HaltedTraverserStrategy extends TraversalStrategy {
    /**
     * @param {String} haltedTraverserFactory full qualified class name in Java of a {@code HaltedTraverserFactory} implementation
     */
    constructor(haltedTraverserFactory) {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.decoration.HaltedTraverserStrategy');
        if (haltedTraverserFactory !== undefined) {
            this.configuration['haltedTraverserFactory'] = haltedTraverserFactory;
        }
    }
}
exports.HaltedTraverserStrategy = HaltedTraverserStrategy;
class OptionsStrategy extends TraversalStrategy {
    constructor(options) {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.decoration.OptionsStrategy', options);
    }
}
exports.OptionsStrategy = OptionsStrategy;
class PartitionStrategy extends TraversalStrategy {
    /**
     * @param {Object} [options]
     * @param {String} [options.partitionKey] name of the property key to partition by
     * @param {String} [options.writePartition] the value of the currently write partition
     * @param {Array<String>} [options.readPartitions] list of strings representing the partitions to include for reads
     * @param {boolean} [options.includeMetaProperties] determines if meta-properties should be included in partitioning defaulting to false
     */
    constructor(options) {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.decoration.PartitionStrategy', options);
    }
}
exports.PartitionStrategy = PartitionStrategy;
class SubgraphStrategy extends TraversalStrategy {
    /**
     * @param {Object} [options]
     * @param {GraphTraversal} [options.vertices] name of the property key to partition by
     * @param {GraphTraversal} [options.edges] the value of the currently write partition
     * @param {GraphTraversal} [options.vertexProperties] list of strings representing the partitions to include for reads
     * @param {boolean} [options.checkAdjacentVertices] enables the strategy to apply the {@code vertices} filter to the adjacent vertices of an edge.
     */
    constructor(options) {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.decoration.SubgraphStrategy', options);
        if (this.configuration.vertices instanceof traversal_js_1.Traversal) {
            this.configuration.vertices = this.configuration.vertices.bytecode;
        }
        if (this.configuration.edges instanceof traversal_js_1.Traversal) {
            this.configuration.edges = this.configuration.edges.bytecode;
        }
        if (this.configuration.vertexProperties instanceof traversal_js_1.Traversal) {
            this.configuration.vertexProperties = this.configuration.vertexProperties.bytecode;
        }
    }
}
exports.SubgraphStrategy = SubgraphStrategy;
class ProductiveByStrategy extends TraversalStrategy {
    /**
     * @param {Object} [options]
     * @param {Array<String>} [options.productiveKeys] set of keys that will always be productive
     */
    constructor(options) {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.ProductiveByStrategy', options);
    }
}
exports.ProductiveByStrategy = ProductiveByStrategy;
class VertexProgramStrategy extends TraversalStrategy {
    constructor(options) {
        super('org.apache.tinkerpop.gremlin.process.computer.traversal.strategy.decoration.VertexProgramStrategy', options);
    }
}
exports.VertexProgramStrategy = VertexProgramStrategy;
class MatchAlgorithmStrategy extends TraversalStrategy {
    /**
     * @param matchAlgorithm
     */
    constructor(matchAlgorithm) {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.finalization.MatchAlgorithmStrategy');
        if (matchAlgorithm !== undefined) {
            this.configuration['matchAlgorithm'] = matchAlgorithm;
        }
    }
}
exports.MatchAlgorithmStrategy = MatchAlgorithmStrategy;
class AdjacentToIncidentStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.AdjacentToIncidentStrategy');
    }
}
exports.AdjacentToIncidentStrategy = AdjacentToIncidentStrategy;
class FilterRankingStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.FilterRankingStrategy');
    }
}
exports.FilterRankingStrategy = FilterRankingStrategy;
class IdentityRemovalStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.IdentityRemovalStrategy');
    }
}
exports.IdentityRemovalStrategy = IdentityRemovalStrategy;
class IncidentToAdjacentStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.IncidentToAdjacentStrategy');
    }
}
exports.IncidentToAdjacentStrategy = IncidentToAdjacentStrategy;
class InlineFilterStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.InlineFilterStrategy');
    }
}
exports.InlineFilterStrategy = InlineFilterStrategy;
class LazyBarrierStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.LazyBarrierStrategy');
    }
}
exports.LazyBarrierStrategy = LazyBarrierStrategy;
class MatchPredicateStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.MatchPredicateStrategy');
    }
}
exports.MatchPredicateStrategy = MatchPredicateStrategy;
class OrderLimitStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.OrderLimitStrategy');
    }
}
exports.OrderLimitStrategy = OrderLimitStrategy;
class PathProcessorStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.PathProcessorStrategy');
    }
}
exports.PathProcessorStrategy = PathProcessorStrategy;
class PathRetractionStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.PathRetractionStrategy');
    }
}
exports.PathRetractionStrategy = PathRetractionStrategy;
class CountStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.CountStrategy');
    }
}
exports.CountStrategy = CountStrategy;
class RepeatUnrollStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.RepeatUnrollStrategy');
    }
}
exports.RepeatUnrollStrategy = RepeatUnrollStrategy;
class GraphFilterStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.GraphFilterStrategy');
    }
}
exports.GraphFilterStrategy = GraphFilterStrategy;
class EarlyLimitStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization.EarlyLimitStrategy');
    }
}
exports.EarlyLimitStrategy = EarlyLimitStrategy;
class LambdaRestrictionStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.verification.LambdaRestrictionStrategy');
    }
}
exports.LambdaRestrictionStrategy = LambdaRestrictionStrategy;
class ReadOnlyStrategy extends TraversalStrategy {
    constructor() {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.verification.ReadOnlyStrategy');
    }
}
exports.ReadOnlyStrategy = ReadOnlyStrategy;
class EdgeLabelVerificationStrategy extends TraversalStrategy {
    /**
     * @param {boolean} logWarnings determines if warnings should be written to the logger when verification fails
     * @param {boolean} throwException determines if exceptions should be thrown when verifications fails
     */
    constructor(logWarnings = false, throwException = false) {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.verification.EdgeLabelVerificationStrategy', {
            logWarnings: logWarnings,
            throwException: throwException,
        });
    }
}
exports.EdgeLabelVerificationStrategy = EdgeLabelVerificationStrategy;
class ReservedKeysVerificationStrategy extends TraversalStrategy {
    /**
     * @param {boolean} logWarnings determines if warnings should be written to the logger when verification fails
     * @param {boolean} throwException determines if exceptions should be thrown when verifications fails
     * @param {Array<String>} keys the list of reserved keys to verify
     */
    constructor(logWarnings = false, throwException = false, keys = ['id', 'label']) {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.verification.ReservedKeysVerificationStrategy', {
            logWarnings: logWarnings,
            throwException: throwException,
            keys: keys,
        });
    }
}
exports.ReservedKeysVerificationStrategy = ReservedKeysVerificationStrategy;
class SeedStrategy extends TraversalStrategy {
    /**
     * @param {SeedStrategyOptions} [options]
     * @param {number} [options.seed] the seed to provide to the random number generator for the traversal
     */
    constructor(options) {
        super('org.apache.tinkerpop.gremlin.process.traversal.strategy.decoration.SeedStrategy', {
            seed: options.seed,
        });
    }
}
exports.SeedStrategy = SeedStrategy;
