/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.tez.serviceplugins.api;


import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.yarn.api.records.ApplicationAttemptId;
import org.apache.hadoop.yarn.api.records.ContainerId;

@InterfaceAudience.Public
@InterfaceStability.Unstable
public interface ContainerLauncherContext extends ServicePluginContextBase {

  // TODO TEZ-2003 (post) TEZ-2664 Tez abstraction for ContainerId, NodeId, other YARN constructs

  // Reporting APIs

  /**
   * Inform the framework that a container has been launched
   *
   * @param containerId the id of the container that has been launched
   */
  void containerLaunched(ContainerId containerId);

  /**
   * Inform the framework of an issue while trying to launch a container.
   *
   * @param containerId the id of the container which failed to launch
   * @param diagnostics diagnostics for the failure
   */
  void containerLaunchFailed(ContainerId containerId, String diagnostics);

  /**
   * Inform the framework that a request has been made to stop a container
   *
   * @param containerId the id of the associated container
   */
  void containerStopRequested(ContainerId containerId);

  /**
   * Inform the framework that the attempt to stop a container failed
   *
   * @param containerId the id of the associated container
   * @param diagnostics diagnostics for the failure
   */
  void containerStopFailed(ContainerId containerId, String diagnostics);

  // TODO TEZ-2003 (post). TEZ-2676 TaskAttemptEndReason does not belong here, and is an unnecessary leak.
  // ContainerCompleted is normally generated by the scheduler in case of YARN since the RM informs about completion.
  // For other sources, there may not be a central entity making this information available. The ContainerLauncher
  // on the stop request will likely be the best place to generate it.

  /**
   * Inform the scheduler that a container was successfully stopped
   *
   * @param containerId the id of the associated container
   * @param exitStatus  the exit status of the container
   * @param diagnostics diagnostics associated with the container end
   * @param endReason   the end reason for the task running in the container
   */
  void containerCompleted(ContainerId containerId, int exitStatus, String diagnostics,
                          TaskAttemptEndReason endReason);

  // Lookup APIs

  /**
   * Get the number of nodes being handled by the specified source
   *
   * @param sourceName the relevant source name
   * @return the number of nodes
   */
  int getNumNodes(String sourceName);

  /**
   * Get the application attempt id for the running application. Relevant when running under YARN
   *
   * @return the applicationAttemptId for the running app
   */
  ApplicationAttemptId getApplicationAttemptId();

  /**
   * Get meta info from the specified TaskCommunicator. This assumes that the launcher has been
   * setup
   * along with a compatible TaskCommunicator, and the launcher knows how to read this meta-info
   *
   * @param taskCommName the name of the task communicator
   * @return meta info for the requested task communicator
   *
   */
  Object getTaskCommunicatorMetaInfo(String taskCommName);

}
