/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tez.dag.history.logging.ats;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.yarn.api.records.timeline.TimelineEntity;
import org.apache.hadoop.yarn.api.records.timeline.TimelinePutResponse;
import org.apache.hadoop.yarn.client.api.TimelineClient;
import org.apache.tez.common.ReflectionUtils;
import org.apache.tez.common.security.DAGAccessControls;
import org.apache.tez.common.security.HistoryACLPolicyException;
import org.apache.tez.common.security.HistoryACLPolicyManager;
import org.apache.tez.dag.api.DagTypeConverters;
import org.apache.tez.dag.api.TezReflectionException;
import org.apache.tez.dag.api.records.DAGProtos;
import org.apache.tez.dag.history.DAGHistoryEvent;
import org.apache.tez.dag.history.HistoryEvent;
import org.apache.tez.dag.history.HistoryEventType;
import org.apache.tez.dag.history.events.DAGRecoveredEvent;
import org.apache.tez.dag.history.events.DAGSubmittedEvent;
import org.apache.tez.dag.history.logging.HistoryLoggingService;
import org.apache.tez.dag.history.logging.ats.HistoryEventTimelineConversion;
import org.apache.tez.dag.records.TezDAGID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ATSHistoryLoggingService
extends HistoryLoggingService {
    private static final Logger LOG = LoggerFactory.getLogger(ATSHistoryLoggingService.class);
    @VisibleForTesting
    LinkedBlockingQueue<DAGHistoryEvent> eventQueue = new LinkedBlockingQueue();
    private Thread eventHandlingThread;
    private AtomicBoolean stopped = new AtomicBoolean(false);
    private int eventCounter = 0;
    private int eventsProcessed = 0;
    private final Object lock = new Object();
    private boolean historyLoggingEnabled = true;
    @VisibleForTesting
    TimelineClient timelineClient;
    private HashSet<TezDAGID> skippedDAGs = new HashSet();
    private Map<TezDAGID, String> dagDomainIdMap = new HashMap<TezDAGID, String>();
    private long maxTimeToWaitOnShutdown;
    private boolean waitForeverOnShutdown = false;
    private int maxEventsPerBatch;
    private long maxPollingTimeMillis;
    private String sessionDomainId;
    private static final String atsHistoryLoggingServiceClassName = "org.apache.tez.dag.history.logging.ats.ATSHistoryLoggingService";
    private static final String atsHistoryACLManagerClassName = "org.apache.tez.dag.history.ats.acls.ATSHistoryACLPolicyManager";
    @VisibleForTesting
    HistoryACLPolicyManager historyACLPolicyManager;

    public ATSHistoryLoggingService() {
        super(ATSHistoryLoggingService.class.getName());
    }

    public void serviceInit(Configuration conf) throws Exception {
        this.historyLoggingEnabled = conf.getBoolean("tez.am.history.logging.enabled", true);
        if (!this.historyLoggingEnabled) {
            LOG.info("ATSService: History Logging disabled. tez.am.history.logging.enabled set to false");
            return;
        }
        if (conf.getBoolean("yarn.timeline-service.enabled", false)) {
            this.timelineClient = TimelineClient.createTimelineClient();
            this.timelineClient.init(conf);
        } else {
            this.timelineClient = null;
            if (conf.get("tez.history.logging.service.class", "").equals(atsHistoryLoggingServiceClassName)) {
                LOG.warn("org.apache.tez.dag.history.logging.ats.ATSHistoryLoggingService is disabled due to Timeline Service being disabled, yarn.timeline-service.enabled set to false");
            }
        }
        this.maxTimeToWaitOnShutdown = conf.getLong("tez.yarn.ats.event.flush.timeout.millis", -1L);
        this.maxEventsPerBatch = conf.getInt("tez.yarn.ats.max.events.per.batch", 5);
        this.maxPollingTimeMillis = conf.getInt("tez.yarn.ats.max.polling.time.per.event.millis", 10);
        if (this.maxTimeToWaitOnShutdown < 0L) {
            this.waitForeverOnShutdown = true;
        }
        LOG.info("Initializing " + ATSHistoryLoggingService.class.getSimpleName() + " with maxEventsPerBatch=" + this.maxEventsPerBatch + ", maxPollingTime(ms)=" + this.maxPollingTimeMillis + ", waitTimeForShutdown(ms)=" + this.maxTimeToWaitOnShutdown + ", TimelineACLManagerClass=" + atsHistoryACLManagerClassName);
        try {
            this.historyACLPolicyManager = (HistoryACLPolicyManager)ReflectionUtils.createClazzInstance((String)atsHistoryACLManagerClassName);
            this.historyACLPolicyManager.setConf(conf);
        }
        catch (TezReflectionException e) {
            LOG.warn("Could not instantiate object for org.apache.tez.dag.history.ats.acls.ATSHistoryACLPolicyManager. ACLs cannot be enforced correctly for history data in Timeline", (Throwable)e);
            if (!conf.getBoolean("tez.allow.disabled.timeline-domains", false)) {
                throw e;
            }
            this.historyACLPolicyManager = null;
        }
    }

    public void serviceStart() {
        if (!this.historyLoggingEnabled || this.timelineClient == null) {
            return;
        }
        this.timelineClient.start();
        try {
            this.sessionDomainId = this.createSessionDomain();
        }
        catch (IOException | HistoryACLPolicyException e) {
            LOG.warn("Could not setup history acls, disabling history logging.", e);
            this.historyLoggingEnabled = false;
            return;
        }
        this.eventHandlingThread = new Thread(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                LinkedList events = new LinkedList();
                boolean interrupted = false;
                while (!(ATSHistoryLoggingService.this.stopped.get() || Thread.currentThread().isInterrupted() || interrupted)) {
                    if (ATSHistoryLoggingService.this.eventCounter != 0 && ATSHistoryLoggingService.this.eventCounter % 1000 == 0) {
                        if (ATSHistoryLoggingService.this.eventsProcessed != 0 && !events.isEmpty()) {
                            LOG.info("Event queue stats, eventsProcessedSinceLastUpdate=" + ATSHistoryLoggingService.this.eventsProcessed + ", eventQueueSize=" + ATSHistoryLoggingService.this.eventQueue.size());
                        }
                        ATSHistoryLoggingService.this.eventCounter = 0;
                        ATSHistoryLoggingService.this.eventsProcessed = 0;
                    } else {
                        ++ATSHistoryLoggingService.this.eventCounter;
                    }
                    Object object = ATSHistoryLoggingService.this.lock;
                    synchronized (object) {
                        try {
                            ATSHistoryLoggingService.this.getEventBatch(events);
                        }
                        catch (InterruptedException e) {
                            interrupted = true;
                        }
                        if (events.isEmpty()) {
                            continue;
                        }
                        ATSHistoryLoggingService.this.eventsProcessed = ATSHistoryLoggingService.this.eventsProcessed + events.size();
                        try {
                            ATSHistoryLoggingService.this.handleEvents(events);
                        }
                        catch (Exception e) {
                            LOG.warn("Error handling events", (Throwable)e);
                        }
                    }
                }
            }
        }, "HistoryEventHandlingThread");
        this.eventHandlingThread.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void serviceStop() {
        LOG.info("Stopping ATSService, eventQueueBacklog=" + this.eventQueue.size());
        this.stopped.set(true);
        if (this.eventHandlingThread != null) {
            this.eventHandlingThread.interrupt();
        }
        Object object = this.lock;
        synchronized (object) {
            if (!this.eventQueue.isEmpty()) {
                LOG.warn("ATSService being stopped, eventQueueBacklog=" + this.eventQueue.size() + ", maxTimeLeftToFlush=" + this.maxTimeToWaitOnShutdown + ", waitForever=" + this.waitForeverOnShutdown);
                long startTime = this.appContext.getClock().getTime();
                long endTime = startTime + this.maxTimeToWaitOnShutdown;
                LinkedList<DAGHistoryEvent> events = new LinkedList<DAGHistoryEvent>();
                while (this.waitForeverOnShutdown || endTime >= this.appContext.getClock().getTime()) {
                    try {
                        this.getEventBatch(events);
                    }
                    catch (InterruptedException e) {
                        LOG.info("ATSService interrupted while shutting down. Exiting. EventQueueBacklog=" + this.eventQueue.size());
                    }
                    if (events.isEmpty()) {
                        LOG.info("Event queue empty, stopping ATS Service");
                        break;
                    }
                    try {
                        this.handleEvents(events);
                    }
                    catch (Exception e) {
                        LOG.warn("Error handling event", (Throwable)e);
                        break;
                    }
                }
            }
        }
        if (!this.eventQueue.isEmpty()) {
            LOG.warn("Did not finish flushing eventQueue before stopping ATSService, eventQueueBacklog=" + this.eventQueue.size());
        }
        if (this.timelineClient != null) {
            this.timelineClient.stop();
        }
        if (this.historyACLPolicyManager != null) {
            this.historyACLPolicyManager.close();
        }
    }

    private void getEventBatch(List<DAGHistoryEvent> events) throws InterruptedException {
        DAGHistoryEvent event;
        events.clear();
        int counter = 0;
        while (counter < this.maxEventsPerBatch && (event = this.eventQueue.poll(this.maxPollingTimeMillis, TimeUnit.MILLISECONDS)) != null) {
            if (!this.isValidEvent(event)) continue;
            ++counter;
            events.add(event);
            if (!event.getHistoryEvent().getEventType().equals((Object)HistoryEventType.DAG_SUBMITTED)) continue;
            break;
        }
    }

    public void handle(DAGHistoryEvent event) {
        if (this.historyLoggingEnabled && this.timelineClient != null) {
            this.eventQueue.add(event);
        }
    }

    private boolean isValidEvent(DAGHistoryEvent event) {
        DAGRecoveredEvent dagRecoveredEvent;
        DAGSubmittedEvent dagSubmittedEvent;
        String dagName;
        HistoryEventType eventType = event.getHistoryEvent().getEventType();
        TezDAGID dagId = event.getDAGID();
        if (eventType.equals((Object)HistoryEventType.DAG_SUBMITTED) && ((dagName = (dagSubmittedEvent = (DAGSubmittedEvent)event.getHistoryEvent()).getDAGName()) != null && dagName.startsWith("TezPreWarmDAG") || !dagSubmittedEvent.isHistoryLoggingEnabled())) {
            this.skippedDAGs.add(dagId);
            return false;
        }
        if (eventType.equals((Object)HistoryEventType.DAG_RECOVERED) && !(dagRecoveredEvent = (DAGRecoveredEvent)event.getHistoryEvent()).isHistoryLoggingEnabled()) {
            this.skippedDAGs.add(dagRecoveredEvent.getDagID());
            return false;
        }
        if (eventType.equals((Object)HistoryEventType.DAG_FINISHED) && this.skippedDAGs.remove(dagId)) {
            return false;
        }
        return dagId == null || !this.skippedDAGs.contains(dagId);
    }

    private void handleEvents(List<DAGHistoryEvent> events) {
        ArrayList<TimelineEntity> entities = new ArrayList<TimelineEntity>(events.size());
        for (DAGHistoryEvent event : events) {
            String domainId = this.getDomainForEvent(event);
            if (event.getDAGID() != null && this.skippedDAGs.contains(event.getDAGID())) continue;
            List<TimelineEntity> eventEntities = HistoryEventTimelineConversion.convertToTimelineEntities(event.getHistoryEvent());
            entities.addAll(eventEntities);
            if (this.historyACLPolicyManager == null || domainId == null || domainId.isEmpty()) continue;
            for (TimelineEntity entity : eventEntities) {
                this.historyACLPolicyManager.updateTimelineEntityDomain((Object)entity, domainId);
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Sending event batch to Timeline, batchSize=" + events.size());
        }
        try {
            TimelinePutResponse response = this.timelineClient.putEntities(entities.toArray(new TimelineEntity[entities.size()]));
            if (response != null && !response.getErrors().isEmpty()) {
                int count = response.getErrors().size();
                for (int i = 0; i < count; ++i) {
                    TimelinePutResponse.TimelinePutError err = (TimelinePutResponse.TimelinePutError)response.getErrors().get(i);
                    if (err.getErrorCode() == 0) continue;
                    LOG.warn("Could not post history event to ATS, atsPutError=" + err.getErrorCode() + ", entityId=" + err.getEntityId());
                }
            }
        }
        catch (Exception e) {
            LOG.warn("Could not handle history events", (Throwable)e);
        }
    }

    private String getDomainForEvent(DAGHistoryEvent event) {
        String domainId = this.sessionDomainId;
        if (this.historyACLPolicyManager == null) {
            return domainId;
        }
        TezDAGID dagId = event.getDAGID();
        HistoryEvent historyEvent = event.getHistoryEvent();
        if (dagId == null || !HistoryEventType.isDAGSpecificEvent((HistoryEventType)historyEvent.getEventType())) {
            return domainId;
        }
        if (this.dagDomainIdMap.containsKey(dagId)) {
            domainId = this.dagDomainIdMap.get(dagId);
            if (historyEvent.getEventType() == HistoryEventType.DAG_FINISHED) {
                this.dagDomainIdMap.remove(dagId);
            }
        } else if (HistoryEventType.DAG_SUBMITTED == historyEvent.getEventType() || HistoryEventType.DAG_RECOVERED == historyEvent.getEventType()) {
            DAGProtos.DAGPlan dagPlan;
            Configuration conf;
            if (HistoryEventType.DAG_SUBMITTED == historyEvent.getEventType()) {
                conf = ((DAGSubmittedEvent)historyEvent).getConf();
                dagPlan = ((DAGSubmittedEvent)historyEvent).getDAGPlan();
            } else {
                conf = this.appContext.getCurrentDAG().getConf();
                dagPlan = this.appContext.getCurrentDAG().getJobPlan();
            }
            domainId = this.createDagDomain(conf, dagPlan, dagId);
            if (this.skippedDAGs.contains(dagId)) {
                return null;
            }
            this.dagDomainIdMap.put(dagId, domainId);
        }
        return domainId;
    }

    private String createSessionDomain() throws HistoryACLPolicyException, IOException {
        if (this.historyACLPolicyManager == null) {
            return null;
        }
        Map domainInfo = this.historyACLPolicyManager.setupSessionACLs(this.getConfig(), this.appContext.getApplicationID());
        if (domainInfo != null) {
            return (String)domainInfo.get("tez.yarn.ats.acl.session.domain.id");
        }
        return null;
    }

    private String createDagDomain(Configuration dagConf, DAGProtos.DAGPlan dagPlan, TezDAGID dagId) {
        if (!this.appContext.isSession()) {
            return this.sessionDomainId;
        }
        DAGAccessControls dagAccessControls = dagPlan.hasAclInfo() ? DagTypeConverters.convertDAGAccessControlsFromProto((DAGProtos.ACLInfo)dagPlan.getAclInfo()) : null;
        try {
            Map domainInfo = this.historyACLPolicyManager.setupSessionDAGACLs(dagConf, this.appContext.getApplicationID(), Integer.toString(dagId.getId()), dagAccessControls);
            if (domainInfo != null) {
                return (String)domainInfo.get("tez.yarn.ats.acl.dag.domain.id");
            }
            return this.sessionDomainId;
        }
        catch (IOException | HistoryACLPolicyException e) {
            LOG.warn("Could not setup ACLs for DAG, disabling history logging for dag.", e);
            this.skippedDAGs.add(dagId);
            return null;
        }
    }
}

