/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.raftlog.segmented;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.LongSupplier;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.metrics.Timekeeper;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.metrics.SegmentedRaftLogMetrics;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.raftlog.LogEntryHeader;
import org.apache.ratis.server.raftlog.LogProtoUtils;
import org.apache.ratis.server.raftlog.RaftLog;
import org.apache.ratis.server.raftlog.RaftLogBase;
import org.apache.ratis.server.raftlog.RaftLogIOException;
import org.apache.ratis.server.raftlog.segmented.LogSegment;
import org.apache.ratis.server.raftlog.segmented.LogSegmentPath;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogCache;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogWorker;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.RaftStorageMetadata;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.AutoCloseableLock;
import org.apache.ratis.util.AwaitToRun;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.StringUtils;
import org.apache.ratis.util.UncheckedAutoCloseable;

public final class SegmentedRaftLog
extends RaftLogBase {
    private final ServerLogMethods server;
    private final RaftStorage storage;
    private final StateMachine stateMachine;
    private final SegmentedRaftLogCache cache;
    private final AwaitToRun cacheEviction;
    private final SegmentedRaftLogWorker fileLogWorker;
    private final long segmentMaxSize;
    private final boolean stateMachineCachingEnabled;
    private final SegmentedRaftLogMetrics metrics;

    private ServerLogMethods newServerLogMethods(final RaftServer.Division impl, final Consumer<RaftProtos.LogEntryProto> notifyTruncatedLogEntry, final BiFunction<RaftProtos.LogEntryProto, Boolean, TransactionContext> getTransactionContext) {
        if (impl == null) {
            return ServerLogMethods.DUMMY;
        }
        return new ServerLogMethods(){

            @Override
            public long[] getFollowerNextIndices() {
                return impl.getInfo().getFollowerNextIndices();
            }

            @Override
            public long getLastAppliedIndex() {
                return impl.getInfo().getLastAppliedIndex();
            }

            @Override
            public void notifyTruncatedLogEntry(TermIndex ti) {
                try {
                    RaftProtos.LogEntryProto entry = SegmentedRaftLog.this.get(ti.getIndex());
                    notifyTruncatedLogEntry.accept(entry);
                }
                catch (RaftLogIOException e) {
                    RaftLog.LOG.error("{}: Failed to read log {}", SegmentedRaftLog.this.getName(), ti, e);
                }
            }

            @Override
            public TransactionContext getTransactionContext(RaftProtos.LogEntryProto entry, boolean createNew) {
                return (TransactionContext)getTransactionContext.apply(entry, createNew);
            }
        };
    }

    private SegmentedRaftLog(Builder b) {
        super(b.memberId, b.snapshotIndexSupplier, b.properties);
        this.metrics = new SegmentedRaftLogMetrics(b.memberId);
        this.server = this.newServerLogMethods(b.server, b.notifyTruncatedLogEntry, b.getTransactionContext);
        this.storage = b.storage;
        this.stateMachine = b.stateMachine;
        this.segmentMaxSize = RaftServerConfigKeys.Log.segmentSizeMax(b.properties).getSize();
        this.cache = new SegmentedRaftLogCache(b.memberId, this.storage, b.properties, this.getRaftLogMetrics());
        this.cacheEviction = new AwaitToRun(b.memberId + "-cacheEviction", this::checkAndEvictCache).start();
        this.fileLogWorker = new SegmentedRaftLogWorker(b.memberId, this.stateMachine, b.submitUpdateCommitEvent, b.server, this.storage, b.properties, this.getRaftLogMetrics());
        this.stateMachineCachingEnabled = RaftServerConfigKeys.Log.StateMachineData.cachingEnabled(b.properties);
    }

    @Override
    public SegmentedRaftLogMetrics getRaftLogMetrics() {
        return this.metrics;
    }

    @Override
    protected void openImpl(long lastIndexInSnapshot, Consumer<RaftProtos.LogEntryProto> consumer) throws IOException {
        this.loadLogSegments(lastIndexInSnapshot, consumer);
        File openSegmentFile = Optional.ofNullable(this.cache.getOpenSegment()).map(LogSegment::getFile).orElse(null);
        this.fileLogWorker.start(Math.max(this.cache.getEndIndex(), lastIndexInSnapshot), Math.min(this.cache.getLastIndexInClosedSegments(), lastIndexInSnapshot), openSegmentFile);
    }

    @Override
    public long getStartIndex() {
        return this.cache.getStartIndex();
    }

    private void loadLogSegments(long lastIndexInSnapshot, Consumer<RaftProtos.LogEntryProto> logConsumer) throws IOException {
        try (AutoCloseableLock writeLock = this.writeLock();){
            List<LogSegmentPath> paths = LogSegmentPath.getLogSegmentPaths(this.storage);
            int i = 0;
            for (LogSegmentPath pi : paths) {
                boolean keepEntryInCache = paths.size() - i++ <= this.cache.getMaxCachedSegments();
                UncheckedAutoCloseable ignored = this.getRaftLogMetrics().startLoadSegmentTimer();
                Throwable throwable = null;
                try {
                    this.cache.loadSegment(pi, keepEntryInCache, logConsumer);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (ignored == null) continue;
                    if (throwable != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    ignored.close();
                }
            }
            if (!this.cache.isEmpty() && this.cache.getEndIndex() < lastIndexInSnapshot) {
                LOG.warn("End log index {} is smaller than last index in snapshot {}", (Object)this.cache.getEndIndex(), (Object)lastIndexInSnapshot);
                this.purgeImpl(lastIndexInSnapshot);
            }
        }
    }

    @Override
    public RaftProtos.LogEntryProto get(long index) throws RaftLogIOException {
        LogSegment.LogRecord record;
        LogSegment segment;
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            segment = this.cache.getSegment(index);
            if (segment == null) {
                RaftProtos.LogEntryProto logEntryProto = null;
                return logEntryProto;
            }
            record = segment.getLogRecord(index);
            if (record == null) {
                RaftProtos.LogEntryProto logEntryProto = null;
                return logEntryProto;
            }
            RaftProtos.LogEntryProto entry = segment.getEntryFromCache(record.getTermIndex());
            if (entry != null) {
                this.getRaftLogMetrics().onRaftLogCacheHit();
                RaftProtos.LogEntryProto logEntryProto = entry;
                return logEntryProto;
            }
        }
        this.getRaftLogMetrics().onRaftLogCacheMiss();
        this.cacheEviction.signal();
        return segment.loadCache(record);
    }

    @Override
    public RaftLog.EntryWithData getEntryWithData(long index) throws RaftLogIOException {
        RaftProtos.LogEntryProto entry = this.get(index);
        if (entry == null) {
            throw new RaftLogIOException("Log entry not found: index = " + index);
        }
        if (!LogProtoUtils.isStateMachineDataEmpty(entry)) {
            return this.newEntryWithData(entry, null);
        }
        try {
            CompletionStage future = null;
            if (this.stateMachine != null) {
                future = this.stateMachine.data().read(entry, this.server.getTransactionContext(entry, false)).exceptionally(ex -> {
                    this.stateMachine.event().notifyLogFailed((Throwable)ex, entry);
                    throw new CompletionException("Failed to read state machine data for log entry " + entry, (Throwable)ex);
                });
            }
            return this.newEntryWithData(entry, (CompletableFuture<ByteString>)future);
        }
        catch (Exception e) {
            String err = this.getName() + ": Failed readStateMachineData for " + LogProtoUtils.toLogEntryString(entry);
            LOG.error(err, e);
            throw new RaftLogIOException(err, JavaUtils.unwrapCompletionException(e));
        }
    }

    private void checkAndEvictCache() {
        if (this.cache.shouldEvict()) {
            this.cache.evictCache(this.server.getFollowerNextIndices(), this.fileLogWorker.getSafeCacheEvictIndex(), this.server.getLastAppliedIndex());
        }
    }

    @Override
    public TermIndex getTermIndex(long index) {
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            LogSegment.LogRecord record = this.cache.getLogRecord(index);
            TermIndex termIndex = record != null ? record.getTermIndex() : null;
            return termIndex;
        }
    }

    @Override
    public LogEntryHeader[] getEntries(long startIndex, long endIndex) {
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            LogEntryHeader[] logEntryHeaderArray = this.cache.getTermIndices(startIndex, endIndex);
            return logEntryHeaderArray;
        }
    }

    @Override
    public TermIndex getLastEntryTermIndex() {
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            TermIndex termIndex = this.cache.getLastTermIndex();
            return termIndex;
        }
    }

    @Override
    protected CompletableFuture<Long> truncateImpl(long index) {
        this.checkLogState();
        try (AutoCloseableLock writeLock = this.writeLock();){
            SegmentedRaftLogCache.TruncationSegments ts = this.cache.truncate(index);
            if (ts != null) {
                Task task = this.fileLogWorker.truncate(ts, index);
                CompletableFuture<Long> completableFuture = task.getFuture();
                return completableFuture;
            }
        }
        return CompletableFuture.completedFuture(index);
    }

    @Override
    protected CompletableFuture<Long> purgeImpl(long index) {
        try (AutoCloseableLock writeLock = this.writeLock();){
            SegmentedRaftLogCache.TruncationSegments ts = this.cache.purge(index);
            this.updateSnapshotIndexFromStateMachine();
            LOG.debug("purging segments:{}", (Object)ts);
            if (ts != null) {
                Task task = this.fileLogWorker.purge(ts);
                CompletableFuture<Long> completableFuture = task.getFuture();
                return completableFuture;
            }
        }
        return CompletableFuture.completedFuture(index);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    protected CompletableFuture<Long> appendEntryImpl(RaftProtos.LogEntryProto entry, TransactionContext context) {
        this.checkLogState();
        if (LOG.isTraceEnabled()) {
            LOG.trace("{}: appendEntry {}", (Object)this.getName(), (Object)LogProtoUtils.toLogEntryString(entry));
        }
        try (AutoCloseableLock writeLock = this.writeLock();){
            Timekeeper.Context appendEntryTimerContext = this.getRaftLogMetrics().startAppendEntryTimer();
            this.validateLogEntry(entry);
            LogSegment currentOpenSegment = this.cache.getOpenSegment();
            boolean rollOpenSegment = false;
            if (currentOpenSegment == null) {
                this.cache.addOpenSegment(entry.getIndex());
                this.fileLogWorker.startLogSegment(entry.getIndex());
            } else if (this.isSegmentFull(currentOpenSegment, entry)) {
                rollOpenSegment = true;
            } else {
                TermIndex last = currentOpenSegment.getLastTermIndex();
                if (last != null && last.getTerm() != entry.getTerm()) {
                    Preconditions.assertTrue(last.getTerm() < entry.getTerm(), "open segment's term %s is larger than the new entry's term %s", last.getTerm(), entry.getTerm());
                    rollOpenSegment = true;
                }
            }
            if (rollOpenSegment) {
                this.cache.rollOpenSegment(true);
                this.fileLogWorker.rollLogSegment(currentOpenSegment);
                this.cacheEviction.signal();
            }
            CompletableFuture<Long> writeFuture = this.fileLogWorker.writeLogEntry(entry, context).getFuture();
            if (this.stateMachineCachingEnabled) {
                this.cache.appendEntry(LogProtoUtils.removeStateMachineData(entry), LogSegment.Op.WRITE_CACHE_WITH_STATE_MACHINE_CACHE);
            } else {
                this.cache.appendEntry(entry, LogSegment.Op.WRITE_CACHE_WITHOUT_STATE_MACHINE_CACHE);
            }
            writeFuture.whenComplete((clientReply, exception) -> appendEntryTimerContext.stop());
            CompletableFuture<Long> completableFuture = writeFuture;
            return completableFuture;
        }
        catch (Exception e) {
            LOG.error("{}: Failed to append {}", this.getName(), LogProtoUtils.toLogEntryString(entry), e);
            throw e;
        }
    }

    private boolean isSegmentFull(LogSegment segment, RaftProtos.LogEntryProto entry) {
        if (segment.getTotalFileSize() >= this.segmentMaxSize) {
            return true;
        }
        long entrySize = LogSegment.getEntrySize(entry, LogSegment.Op.CHECK_SEGMENT_FILE_FULL);
        return entrySize <= this.segmentMaxSize && segment.getTotalFileSize() + entrySize > this.segmentMaxSize;
    }

    @Override
    public List<CompletableFuture<Long>> appendImpl(List<RaftProtos.LogEntryProto> entries) {
        this.checkLogState();
        if (entries == null || entries.isEmpty()) {
            return Collections.emptyList();
        }
        try (AutoCloseableLock writeLock = this.writeLock();){
            ArrayList<CompletableFuture<Long>> futures;
            SegmentedRaftLogCache.TruncateIndices ti = this.cache.computeTruncateIndices(this.server::notifyTruncatedLogEntry, entries);
            long truncateIndex = ti.getTruncateIndex();
            int index = ti.getArrayIndex();
            LOG.debug("truncateIndex={}, arrayIndex={}", (Object)truncateIndex, (Object)index);
            if (truncateIndex != -1L) {
                futures = new ArrayList(entries.size() - index + 1);
                futures.add(this.truncate(truncateIndex));
            } else {
                futures = new ArrayList<CompletableFuture<Long>>(entries.size() - index);
            }
            for (int i = index; i < entries.size(); ++i) {
                RaftProtos.LogEntryProto entry = entries.get(i);
                futures.add(this.appendEntry(entry, this.server.getTransactionContext(entry, true)));
            }
            ArrayList<CompletableFuture<Long>> arrayList = futures;
            return arrayList;
        }
    }

    @Override
    public long getFlushIndex() {
        return this.fileLogWorker.getFlushIndex();
    }

    @Override
    public void persistMetadata(RaftStorageMetadata metadata) throws IOException {
        this.storage.getMetadataFile().persist(metadata);
    }

    @Override
    public RaftStorageMetadata loadMetadata() throws IOException {
        return this.storage.getMetadataFile().getMetadata();
    }

    @Override
    public CompletableFuture<Long> onSnapshotInstalled(long lastSnapshotIndex) {
        this.updateSnapshotIndex(lastSnapshotIndex);
        this.fileLogWorker.syncWithSnapshot(lastSnapshotIndex);
        LogSegment openSegment = this.cache.getOpenSegment();
        if (openSegment != null && openSegment.hasEntries()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("syncWithSnapshot : Found open segment {}, with end index {}, snapshotIndex {}", openSegment, openSegment.getEndIndex(), lastSnapshotIndex);
            }
            if (openSegment.getEndIndex() <= lastSnapshotIndex) {
                this.fileLogWorker.closeLogSegment(openSegment);
                this.cache.rollOpenSegment(false);
                this.cacheEviction.signal();
            }
        }
        return this.purgeImpl(lastSnapshotIndex);
    }

    @Override
    public void close() throws IOException {
        try (AutoCloseableLock writeLock = this.writeLock();){
            super.close();
            this.cacheEviction.close();
            this.cache.close();
        }
        this.fileLogWorker.close();
        this.storage.close();
        this.getRaftLogMetrics().unregister();
    }

    SegmentedRaftLogCache getRaftLogCache() {
        return this.cache;
    }

    @Override
    public String toLogEntryString(RaftProtos.LogEntryProto logEntry) {
        return LogProtoUtils.toLogEntryString(logEntry, this.stateMachine::toStateMachineLogEntryString);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static final class Builder {
        private RaftGroupMemberId memberId;
        private RaftServer.Division server;
        private StateMachine stateMachine;
        private Consumer<RaftProtos.LogEntryProto> notifyTruncatedLogEntry;
        private BiFunction<RaftProtos.LogEntryProto, Boolean, TransactionContext> getTransactionContext;
        private Runnable submitUpdateCommitEvent;
        private RaftStorage storage;
        private LongSupplier snapshotIndexSupplier = () -> -1L;
        private RaftProperties properties;

        private Builder() {
        }

        public Builder setMemberId(RaftGroupMemberId memberId) {
            this.memberId = memberId;
            return this;
        }

        public Builder setServer(RaftServer.Division server) {
            this.server = server;
            this.stateMachine = server.getStateMachine();
            return this;
        }

        public Builder setStateMachine(StateMachine stateMachine) {
            this.stateMachine = stateMachine;
            return this;
        }

        public Builder setNotifyTruncatedLogEntry(Consumer<RaftProtos.LogEntryProto> notifyTruncatedLogEntry) {
            this.notifyTruncatedLogEntry = notifyTruncatedLogEntry;
            return this;
        }

        public Builder setGetTransactionContext(BiFunction<RaftProtos.LogEntryProto, Boolean, TransactionContext> getTransactionContext) {
            this.getTransactionContext = getTransactionContext;
            return this;
        }

        public Builder setSubmitUpdateCommitEvent(Runnable submitUpdateCommitEvent) {
            this.submitUpdateCommitEvent = submitUpdateCommitEvent;
            return this;
        }

        public Builder setStorage(RaftStorage storage) {
            this.storage = storage;
            return this;
        }

        public Builder setSnapshotIndexSupplier(LongSupplier snapshotIndexSupplier) {
            this.snapshotIndexSupplier = snapshotIndexSupplier;
            return this;
        }

        public Builder setProperties(RaftProperties properties) {
            this.properties = properties;
            return this;
        }

        public SegmentedRaftLog build() {
            return new SegmentedRaftLog(this);
        }
    }

    static interface ServerLogMethods {
        public static final ServerLogMethods DUMMY = new ServerLogMethods(){};

        default public long[] getFollowerNextIndices() {
            return null;
        }

        default public long getLastAppliedIndex() {
            return -1L;
        }

        default public void notifyTruncatedLogEntry(TermIndex ti) {
        }

        default public TransactionContext getTransactionContext(RaftProtos.LogEntryProto entry, boolean createNew) {
            return null;
        }
    }

    static abstract class Task {
        private final CompletableFuture<Long> future = new CompletableFuture();
        private Timekeeper.Context queueTimerContext;

        Task() {
        }

        CompletableFuture<Long> getFuture() {
            return this.future;
        }

        void done() {
            this.completeFuture();
        }

        final void completeFuture() {
            boolean completed = this.future.complete(this.getEndIndex());
            Preconditions.assertTrue(completed, () -> this + " is already " + StringUtils.completableFuture2String(this.future, false));
        }

        void failed(IOException e) {
            this.getFuture().completeExceptionally(e);
        }

        abstract void execute() throws IOException;

        abstract long getEndIndex();

        void startTimerOnEnqueue(Timekeeper queueTimer) {
            this.queueTimerContext = queueTimer.time();
        }

        void stopTimerOnDequeue() {
            if (this.queueTimerContext != null) {
                this.queueTimerContext.stop();
            }
        }

        int getSerializedSize() {
            return 0;
        }

        public String toString() {
            return JavaUtils.getClassSimpleName(this.getClass()) + ":" + this.getEndIndex();
        }
    }
}

