/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.fs;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.time.Duration;
import java.util.OptionalLong;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.hdds.fs.SpaceUsageCheckParams;
import org.apache.hadoop.hdds.fs.SpaceUsagePersistence;
import org.apache.hadoop.hdds.fs.SpaceUsageSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class CachingSpaceUsageSource
implements SpaceUsageSource {
    private static final Logger LOG = LoggerFactory.getLogger(CachingSpaceUsageSource.class);
    private final ScheduledExecutorService executor;
    private final AtomicLong cachedValue = new AtomicLong();
    private final Duration refresh;
    private final SpaceUsageSource source;
    private final SpaceUsagePersistence persistence;
    private boolean running;
    private ScheduledFuture<?> scheduledFuture;
    private final AtomicBoolean isRefreshRunning;

    public CachingSpaceUsageSource(SpaceUsageCheckParams params) {
        this(params, CachingSpaceUsageSource.createExecutor(params));
    }

    CachingSpaceUsageSource(SpaceUsageCheckParams params, ScheduledExecutorService executor) {
        Preconditions.checkArgument((params != null ? 1 : 0) != 0, (Object)"params == null");
        this.refresh = params.getRefresh();
        this.source = params.getSource();
        this.persistence = params.getPersistence();
        this.executor = executor;
        this.isRefreshRunning = new AtomicBoolean();
        Preconditions.checkArgument((this.refresh.isZero() == (executor == null) ? 1 : 0) != 0, (Object)"executor should be provided if and only if refresh is requested");
        this.loadInitialValue();
    }

    @Override
    public long getCapacity() {
        return this.source.getCapacity();
    }

    @Override
    public long getAvailable() {
        return this.source.getAvailable();
    }

    @Override
    public long getUsedSpace() {
        return this.cachedValue.get();
    }

    public void incrementUsedSpace(long usedSpace) {
        this.cachedValue.addAndGet(usedSpace);
    }

    public void decrementUsedSpace(long reclaimedSpace) {
        this.cachedValue.addAndGet(-1L * reclaimedSpace);
    }

    public void start() {
        if (this.executor != null) {
            long initialDelay;
            long l = initialDelay = this.cachedValue.get() > 0L ? this.refresh.toMillis() : 0L;
            if (!this.running) {
                this.scheduledFuture = this.executor.scheduleWithFixedDelay(this::refresh, initialDelay, this.refresh.toMillis(), TimeUnit.MILLISECONDS);
                this.running = true;
            }
        } else {
            this.refresh();
        }
    }

    public void shutdown() {
        this.persistence.save(this);
        if (this.executor != null) {
            if (this.running && this.scheduledFuture != null) {
                this.scheduledFuture.cancel(true);
            }
            this.running = false;
            this.executor.shutdown();
        }
    }

    public void refreshNow() {
        this.executor.schedule(this::refresh, 0L, TimeUnit.MILLISECONDS);
    }

    private void loadInitialValue() {
        OptionalLong initialValue = this.persistence.load();
        initialValue.ifPresent(this.cachedValue::set);
    }

    private void refresh() {
        if (this.isRefreshRunning.compareAndSet(false, true)) {
            try {
                this.cachedValue.set(this.source.getUsedSpace());
            }
            catch (RuntimeException e) {
                LOG.warn("Error refreshing space usage for {}", (Object)this.source, (Object)e);
            }
            finally {
                this.isRefreshRunning.set(false);
            }
        }
    }

    @Nullable
    private static ScheduledExecutorService createExecutor(SpaceUsageCheckParams params) {
        if (params.getRefresh().isZero()) {
            return null;
        }
        return Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("DiskUsage-" + params.getPath() + "-%n").build());
    }
}

