/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.commons.resourcehandler.webapp.config.impl;

import java.io.IOException;
import java.net.URL;

import javax.faces.application.ProjectStage;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.servlet.ServletContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.myfaces.commons.resourcehandler.webapp.config.WebConfigProvider;
import org.apache.myfaces.commons.resourcehandler.webapp.config.WebConfigProviderFactory;
import org.apache.myfaces.commons.resourcehandler.webapp.config.WebRegistration;
import org.apache.myfaces.commons.util.WebConfigParamUtils;

/**
 * 
 * @author Leonardo Uribe
 *
 */
public class WebConfigProviderImpl extends WebConfigProvider
{
    private static final String WEB_XML_PATH = "/WEB-INF/web.xml";

    private static final String WEB_REGISTRATION_KEY = WebRegistration.class.getName();
    
    private static final String INIT_PARAM_CONFIG_REFRESH_PERIOD = "org.apache.myfaces.CONFIG_REFRESH_PERIOD";
    private static final long INIT_PARAM_CONFIG_REFRESH_PERIOD_DEFAULT = 2;
    
    private long refreshPeriod = -1;
    
    private long parsingTime = -1;
    
    private Boolean servlet30Mode;
    
    public WebConfigProviderImpl()
    {
    }

    public WebRegistration getWebRegistration(FacesContext context)
    {
        WebRegistration webConfig
                = (WebRegistration)context.getExternalContext().getApplicationMap().get(WEB_REGISTRATION_KEY);
        if (webConfig == null)
        {
            init(context);
            webConfig = (WebRegistration)context.getExternalContext().getApplicationMap().get(WEB_REGISTRATION_KEY);
        }
        return webConfig;
    }

    public void init(FacesContext context)
    {
        if (servlet30Mode == null)
        {
            servlet30Mode = (context.getExternalContext().getContext() instanceof ServletContext
                            && _ExternalSpecifications.isServlet30Available());
        }
        if (servlet30Mode)
        {
            ServletContext servletContext = (ServletContext) context.getExternalContext().getContext(); 
            context.getExternalContext().getApplicationMap().put(WEB_REGISTRATION_KEY,
                    _Servlet30Utils.getWebRegistrationFromServlet30Api(servletContext));
        }
        else
        {
            //Load the information parsing web.xml
            context.getExternalContext().getApplicationMap().put(WEB_REGISTRATION_KEY,
                    WebXmlConfigParser.getWebRegistrationFromParsedWebXml(context));
        }
        if (!context.isProjectStage(ProjectStage.Production))
        {
            long configRefreshPeriod = WebConfigParamUtils.getLongInitParameter(context.getExternalContext(), 
                    INIT_PARAM_CONFIG_REFRESH_PERIOD, INIT_PARAM_CONFIG_REFRESH_PERIOD_DEFAULT);
            parsingTime = System.currentTimeMillis();
            refreshPeriod = (configRefreshPeriod * 1000);
        }
    }

    public void update(FacesContext context)
    {
        if (servlet30Mode == null)
        {
            servlet30Mode = (context.getExternalContext().getContext() instanceof ServletContext
                            && _ExternalSpecifications.isServlet30Available());
        }
        if (!servlet30Mode  && !context.isProjectStage(ProjectStage.Production))
        {
            if (isOld(context.getExternalContext()))
            {
                init(context);
            }
        }
    }
    
    private boolean isOld(ExternalContext context)
    {
        if (refreshPeriod > 0)
        {
            long ttl = this.parsingTime + refreshPeriod;
            if (System.currentTimeMillis() > ttl)
            {
                long lastModified = getWebXmlLastModified(context);
                return lastModified == 0 || lastModified > ttl;
            }
        }
        return false;
    }

    
    private static long getWebXmlLastModified(ExternalContext context)
    {
        try
        {
            URL url = context.getResource(WEB_XML_PATH);
            if (url != null)
            {
                return url.openConnection().getLastModified();
            }
        }
        catch (IOException e)
        {
            Log log = LogFactory.getLog(WebConfigProviderFactory.class);
            if (log.isErrorEnabled())
            {
                log.error("Could not find web.xml in path " + WEB_XML_PATH);
            }
        }
        return 0L;
    }
}
