/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.consumer.internals;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.clients.ApiVersions;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.clients.NodeApiVersions;
import org.apache.kafka.clients.consumer.OffsetAndTimestamp;
import org.apache.kafka.clients.consumer.OffsetResetStrategy;
import org.apache.kafka.clients.consumer.internals.ConsumerMetadata;
import org.apache.kafka.clients.consumer.internals.SubscriptionState;
import org.apache.kafka.common.IsolationLevel;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.TopicAuthorizationException;
import org.apache.kafka.common.message.ApiVersionsResponseData;
import org.apache.kafka.common.message.ListOffsetsRequestData;
import org.apache.kafka.common.message.ListOffsetsResponseData;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.ListOffsetsResponse;
import org.apache.kafka.common.requests.OffsetsForLeaderEpochRequest;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;

class OffsetFetcherUtils {
    private final ConsumerMetadata metadata;
    private final SubscriptionState subscriptionState;
    private final Time time;
    private final long retryBackoffMs;
    private final ApiVersions apiVersions;
    private final Logger log;
    private final AtomicReference<RuntimeException> cachedOffsetForLeaderException = new AtomicReference();
    private final AtomicReference<RuntimeException> cachedListOffsetsException = new AtomicReference();
    private final AtomicInteger metadataUpdateVersion = new AtomicInteger(-1);

    OffsetFetcherUtils(LogContext logContext, ConsumerMetadata metadata, SubscriptionState subscriptionState, Time time, long retryBackoffMs, ApiVersions apiVersions) {
        this.log = logContext.logger(this.getClass());
        this.metadata = metadata;
        this.subscriptionState = subscriptionState;
        this.time = time;
        this.retryBackoffMs = retryBackoffMs;
        this.apiVersions = apiVersions;
    }

    ListOffsetResult handleListOffsetResponse(ListOffsetsResponse listOffsetsResponse) {
        HashMap<TopicPartition, ListOffsetData> fetchedOffsets = new HashMap<TopicPartition, ListOffsetData>();
        HashSet<TopicPartition> partitionsToRetry = new HashSet<TopicPartition>();
        HashSet<String> unauthorizedTopics = new HashSet<String>();
        for (ListOffsetsResponseData.ListOffsetsTopicResponse topic : listOffsetsResponse.topics()) {
            block8: for (ListOffsetsResponseData.ListOffsetsPartitionResponse partition : topic.partitions()) {
                TopicPartition topicPartition = new TopicPartition(topic.name(), partition.partitionIndex());
                Errors error = Errors.forCode(partition.errorCode());
                switch (error) {
                    case NONE: {
                        if (!partition.oldStyleOffsets().isEmpty()) {
                            if (partition.oldStyleOffsets().size() > 1) {
                                throw new IllegalStateException("Unexpected partitionData response of length " + partition.oldStyleOffsets().size());
                            }
                            long offset = partition.oldStyleOffsets().get(0);
                            this.log.debug("Handling v0 ListOffsetResponse response for {}. Fetched offset {}", (Object)topicPartition, (Object)offset);
                            if (offset == -1L) continue block8;
                            ListOffsetData offsetData = new ListOffsetData(offset, null, Optional.empty());
                            fetchedOffsets.put(topicPartition, offsetData);
                            break;
                        }
                        this.log.debug("Handling ListOffsetResponse response for {}. Fetched offset {}, timestamp {}", new Object[]{topicPartition, partition.offset(), partition.timestamp()});
                        if (partition.offset() == -1L) continue block8;
                        Optional<Integer> leaderEpoch = partition.leaderEpoch() == -1 ? Optional.empty() : Optional.of(partition.leaderEpoch());
                        ListOffsetData offsetData = new ListOffsetData(partition.offset(), partition.timestamp(), leaderEpoch);
                        fetchedOffsets.put(topicPartition, offsetData);
                        break;
                    }
                    case UNSUPPORTED_FOR_MESSAGE_FORMAT: {
                        this.log.debug("Cannot search by timestamp for partition {} because the message format version is before 0.10.0", (Object)topicPartition);
                        break;
                    }
                    case NOT_LEADER_OR_FOLLOWER: 
                    case REPLICA_NOT_AVAILABLE: 
                    case KAFKA_STORAGE_ERROR: 
                    case OFFSET_NOT_AVAILABLE: 
                    case LEADER_NOT_AVAILABLE: 
                    case FENCED_LEADER_EPOCH: 
                    case UNKNOWN_LEADER_EPOCH: {
                        this.log.debug("Attempt to fetch offsets for partition {} failed due to {}, retrying.", (Object)topicPartition, (Object)error);
                        partitionsToRetry.add(topicPartition);
                        break;
                    }
                    case UNKNOWN_TOPIC_OR_PARTITION: {
                        this.log.warn("Received unknown topic or partition error in ListOffset request for partition {}", (Object)topicPartition);
                        partitionsToRetry.add(topicPartition);
                        break;
                    }
                    case TOPIC_AUTHORIZATION_FAILED: {
                        unauthorizedTopics.add(topicPartition.topic());
                        break;
                    }
                    default: {
                        this.log.warn("Attempt to fetch offsets for partition {} failed due to unexpected exception: {}, retrying.", (Object)topicPartition, (Object)error.message());
                        partitionsToRetry.add(topicPartition);
                    }
                }
            }
        }
        if (!unauthorizedTopics.isEmpty()) {
            throw new TopicAuthorizationException(unauthorizedTopics);
        }
        return new ListOffsetResult(fetchedOffsets, partitionsToRetry);
    }

    <T> Map<Node, Map<TopicPartition, T>> regroupPartitionMapByNode(Map<TopicPartition, T> partitionMap) {
        return partitionMap.entrySet().stream().collect(Collectors.groupingBy(entry -> this.metadata.fetch().leaderFor((TopicPartition)entry.getKey()), Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
    }

    Map<TopicPartition, SubscriptionState.FetchPosition> getPartitionsToValidate() {
        RuntimeException exception = this.cachedOffsetForLeaderException.getAndSet(null);
        if (exception != null) {
            throw exception;
        }
        this.validatePositionsOnMetadataChange();
        return this.subscriptionState.partitionsNeedingValidation(this.time.milliseconds()).stream().filter(tp -> this.subscriptionState.position((TopicPartition)tp) != null).collect(Collectors.toMap(Function.identity(), this.subscriptionState::position));
    }

    void maybeSetOffsetForLeaderException(RuntimeException e) {
        if (!this.cachedOffsetForLeaderException.compareAndSet(null, e)) {
            this.log.error("Discarding error in OffsetsForLeaderEpoch because another error is pending", (Throwable)e);
        }
    }

    void validatePositionsOnMetadataChange() {
        int newMetadataUpdateVersion = this.metadata.updateVersion();
        if (this.metadataUpdateVersion.getAndSet(newMetadataUpdateVersion) != newMetadataUpdateVersion) {
            this.subscriptionState.assignedPartitions().forEach(topicPartition -> {
                Metadata.LeaderAndEpoch leaderAndEpoch = this.metadata.currentLeader((TopicPartition)topicPartition);
                this.subscriptionState.maybeValidatePositionForCurrentLeader(this.apiVersions, (TopicPartition)topicPartition, leaderAndEpoch);
            });
        }
    }

    Map<TopicPartition, Long> getOffsetResetTimestamp() {
        RuntimeException exception = this.cachedListOffsetsException.getAndSet(null);
        if (exception != null) {
            throw exception;
        }
        Set<TopicPartition> partitions = this.subscriptionState.partitionsNeedingReset(this.time.milliseconds());
        HashMap<TopicPartition, Long> offsetResetTimestamps = new HashMap<TopicPartition, Long>();
        for (TopicPartition partition : partitions) {
            Long timestamp = this.offsetResetStrategyTimestamp(partition);
            if (timestamp == null) continue;
            offsetResetTimestamps.put(partition, timestamp);
        }
        return offsetResetTimestamps;
    }

    static Map<TopicPartition, OffsetAndTimestamp> buildOffsetsForTimesResult(Map<TopicPartition, Long> timestampsToSearch, Map<TopicPartition, ListOffsetData> fetchedOffsets) {
        HashMap<TopicPartition, OffsetAndTimestamp> offsetsByTimes = new HashMap<TopicPartition, OffsetAndTimestamp>(timestampsToSearch.size());
        for (Map.Entry<TopicPartition, Long> entry : timestampsToSearch.entrySet()) {
            offsetsByTimes.put(entry.getKey(), null);
        }
        for (Map.Entry<TopicPartition, Object> entry : fetchedOffsets.entrySet()) {
            ListOffsetData offsetData = (ListOffsetData)entry.getValue();
            offsetsByTimes.put(entry.getKey(), new OffsetAndTimestamp(offsetData.offset, offsetData.timestamp, offsetData.leaderEpoch));
        }
        return offsetsByTimes;
    }

    private Long offsetResetStrategyTimestamp(TopicPartition partition) {
        OffsetResetStrategy strategy = this.subscriptionState.resetStrategy(partition);
        if (strategy == OffsetResetStrategy.EARLIEST) {
            return -2L;
        }
        if (strategy == OffsetResetStrategy.LATEST) {
            return -1L;
        }
        return null;
    }

    static Set<String> topicsForPartitions(Collection<TopicPartition> partitions) {
        return partitions.stream().map(TopicPartition::topic).collect(Collectors.toSet());
    }

    void updateSubscriptionState(Map<TopicPartition, ListOffsetData> fetchedOffsets, IsolationLevel isolationLevel) {
        for (Map.Entry<TopicPartition, ListOffsetData> entry : fetchedOffsets.entrySet()) {
            TopicPartition partition = entry.getKey();
            if (!this.subscriptionState.isAssigned(partition)) continue;
            long offset = entry.getValue().offset;
            if (isolationLevel == IsolationLevel.READ_COMMITTED) {
                this.log.trace("Updating last stable offset for partition {} to {}", (Object)partition, (Object)offset);
                this.subscriptionState.updateLastStableOffset(partition, offset);
                continue;
            }
            this.log.trace("Updating high watermark for partition {} to {}", (Object)partition, (Object)offset);
            this.subscriptionState.updateHighWatermark(partition, offset);
        }
    }

    static OffsetResetStrategy timestampToOffsetResetStrategy(long timestamp) {
        if (timestamp == -2L) {
            return OffsetResetStrategy.EARLIEST;
        }
        if (timestamp == -1L) {
            return OffsetResetStrategy.LATEST;
        }
        return null;
    }

    void onSuccessfulRequestForResettingPositions(Map<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition> resetTimestamps, ListOffsetResult result) {
        if (!result.partitionsToRetry.isEmpty()) {
            this.subscriptionState.requestFailed(result.partitionsToRetry, this.time.milliseconds() + this.retryBackoffMs);
            this.metadata.requestUpdate();
        }
        for (Map.Entry<TopicPartition, ListOffsetData> fetchedOffset : result.fetchedOffsets.entrySet()) {
            TopicPartition partition = fetchedOffset.getKey();
            ListOffsetData offsetData = fetchedOffset.getValue();
            ListOffsetsRequestData.ListOffsetsPartition requestedReset = resetTimestamps.get(partition);
            this.resetPositionIfNeeded(partition, OffsetFetcherUtils.timestampToOffsetResetStrategy(requestedReset.timestamp()), offsetData);
        }
    }

    void onFailedRequestForResettingPositions(Map<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition> resetTimestamps, RuntimeException error) {
        this.subscriptionState.requestFailed(resetTimestamps.keySet(), this.time.milliseconds() + this.retryBackoffMs);
        this.metadata.requestUpdate();
        if (!(error instanceof RetriableException) && !this.cachedListOffsetsException.compareAndSet(null, error)) {
            this.log.error("Discarding error in ListOffsetResponse because another error is pending", (Throwable)error);
        }
    }

    void resetPositionIfNeeded(TopicPartition partition, OffsetResetStrategy requestedResetStrategy, ListOffsetData offsetData) {
        SubscriptionState.FetchPosition position = new SubscriptionState.FetchPosition(offsetData.offset, Optional.empty(), this.metadata.currentLeader(partition));
        offsetData.leaderEpoch.ifPresent(epoch -> this.metadata.updateLastSeenEpochIfNewer(partition, (int)epoch));
        this.subscriptionState.maybeSeekUnvalidated(partition, position, requestedResetStrategy);
    }

    static Map<Node, Map<TopicPartition, SubscriptionState.FetchPosition>> regroupFetchPositionsByLeader(Map<TopicPartition, SubscriptionState.FetchPosition> partitionMap) {
        return partitionMap.entrySet().stream().filter(entry -> ((SubscriptionState.FetchPosition)entry.getValue()).currentLeader.leader.isPresent()).collect(Collectors.groupingBy(entry -> ((SubscriptionState.FetchPosition)entry.getValue()).currentLeader.leader.get(), Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
    }

    static boolean hasUsableOffsetForLeaderEpochVersion(NodeApiVersions nodeApiVersions) {
        ApiVersionsResponseData.ApiVersion apiVersion = nodeApiVersions.apiVersion(ApiKeys.OFFSET_FOR_LEADER_EPOCH);
        if (apiVersion == null) {
            return false;
        }
        return OffsetsForLeaderEpochRequest.supportsTopicPermission(apiVersion.maxVersion());
    }

    static class ListOffsetData {
        final long offset;
        final Long timestamp;
        final Optional<Integer> leaderEpoch;

        ListOffsetData(long offset, Long timestamp, Optional<Integer> leaderEpoch) {
            this.offset = offset;
            this.timestamp = timestamp;
            this.leaderEpoch = leaderEpoch;
        }
    }

    static class ListOffsetResult {
        final Map<TopicPartition, ListOffsetData> fetchedOffsets;
        final Set<TopicPartition> partitionsToRetry;

        ListOffsetResult(Map<TopicPartition, ListOffsetData> fetchedOffsets, Set<TopicPartition> partitionsNeedingRetry) {
            this.fetchedOffsets = fetchedOffsets;
            this.partitionsToRetry = partitionsNeedingRetry;
        }

        ListOffsetResult() {
            this.fetchedOffsets = new HashMap<TopicPartition, ListOffsetData>();
            this.partitionsToRetry = new HashSet<TopicPartition>();
        }
    }
}

