/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#ifndef MOD_FTP_H
#define MOD_FTP_H

#include "apr_general.h" /* stringify */

#define MODFTP_COPYRIGHT \
  "Copyright 2009 The Apache Software Foundation."

#define MODFTP_VERSION_MAJOR  0
#define MODFTP_VERSION_MINOR  9
#define MODFTP_VERSION_SUBVER 6
#define MODFTP_VERSION_DEV    0

#if MODFTP_VERSION_DEV
#define MODFTP_VERSION_DEVSTR "-dev"
#else
#define MODFTP_VERSION_DEVSTR ""
#endif

#define MODFTP_REVISION      APR_STRINGIFY(MODFTP_VERSION_MAJOR) \
                         "." APR_STRINGIFY(MODFTP_VERSION_MINOR) \
                         "." APR_STRINGIFY(MODFTP_VERSION_SUBVER)
#define MODFTP_VERSION       MODFTP_REVISION MODFTP_VERSION_DEVSTR

#define MODFTP_PRODUCT       "mod_ftp/" MODFTP_VERSION

#ifndef VERSION_ONLY

#include "httpd.h"
#include "http_config.h"

#ifdef __cplusplus
extern "C" {
#endif


/* FTP Reply Codes */

#define FTP_REPLY_RESTART_MARKER          110
#define FTP_REPLY_SERVICE_READY_IN_N_MIN  120
#define FTP_REPLY_DATA_ALREADY_OPEN       125
#define FTP_REPLY_FILE_STATUS_OK          150
#define FTP_REPLY_COMMAND_OK              200
#define FTP_REPLY_NOT_IMPLEMENTED         202
#define FTP_REPLY_SYSTEM_STATUS           211
#define FTP_REPLY_DIRECTORY_STATUS        212
#define FTP_REPLY_FILE_STATUS             213
#define FTP_REPLY_HELP_MESSAGE            214
#define FTP_REPLY_SYSTEM_TYPE             215
#define FTP_REPLY_SERVICE_READY           220
#define FTP_REPLY_CONTROL_CLOSE           221
#define FTP_REPLY_DATA_OPEN               225
#define FTP_REPLY_DATA_CLOSE              226
#define FTP_REPLY_PASSIVE_MODE            227
#define FTP_REPLY_EXTENDED_PASSIVE_MODE   229
#define FTP_REPLY_USER_LOGGED_IN          230
#define FTP_REPLY_USER_LOGGED_IN_SECURE   232
#define FTP_REPLY_SECURITY_EXCHANGE_DONE  234
#define FTP_REPLY_COMPLETED               250
#define FTP_REPLY_PATH_CREATED            257
#define FTP_REPLY_USER_OK                 331
#define FTP_REPLY_NEED_ACCOUNT            332
#define FTP_REPLY_PENDING                 350
#define FTP_REPLY_SERVICE_NOT_AVAILABLE   421
#define FTP_REPLY_CANNOT_OPEN_DATACONN    425
#define FTP_REPLY_TRANSFER_ABORTED        426
#define FTP_REPLY_FILE_BUSY               450
#define FTP_REPLY_LOCAL_ERROR             451
#define FTP_REPLY_INSUFFICIENT_SPACE      452
#define FTP_REPLY_COMMAND_UNRECOGNIZED    500
#define FTP_REPLY_SYNTAX_ERROR            501
#define FTP_REPLY_COMMAND_NOT_IMPLEMENTED 502
#define FTP_REPLY_BAD_SEQUENCE            503
#define FTP_REPLY_COMMAND_NOT_IMPL_PARAM  504
#define FTP_REPLY_BAD_PROTOCOL            522
#define FTP_REPLY_NOT_LOGGED_IN           530
#define FTP_REPLY_NEED_ACCOUNT_TO_STORE   532
#define FTP_REPLY_PROT_NOT_SUPPORTED      536
#define FTP_REPLY_FILE_NOT_FOUND          550
#define FTP_REPLY_PAGE_TYPE_UNKNOWN       551
#define FTP_REPLY_EXEEDED_STORAGE         552
#define FTP_REPLY_FILENAME_NOT_ALLOWED    553
#define FTP_REPLY_INVALID_REST_PARAM      554
#define FTP_REPLY_TYPE_OR_STRU_MISMATCH   555

#if defined(WIN32) && defined(FTP_DECLARE_EXPORT)
#define FTP_DECLARE_DATA __declspec(dllexport)
#define FTP_DECLARE(fn)  __declspec(dllexport) fn
#elif defined(WIN32)
#define FTP_DECLARE_DATA __declspec(dllimport)
#define FTP_DECLARE(fn)  __declspec(dllimport) fn
#else
#define FTP_DECLARE_DATA 
#define FTP_DECLARE(fn)  fn
#endif

/* Must be available to reach ftp configuration structures */
extern module FTP_DECLARE_DATA ftp_module;

/* The main ftp server configuration */
typedef struct ftp_server_config
{
    int enabled;
    int options;
    int implicit_ssl;
    int timeout_login;
    int timeout_idle;
    int timeout_data;
    int active_min;
    int active_max;
    int pasv_min;
    int pasv_max;
    char *pasv_addr;
    char *pasv_bindaddr;
    int pasv_bindfamily;
    int epsv_ignore_family;
    int data_block_size;
    const char *banner_message;
    int banner_message_isfile;
    const char *exit_message;
    int exit_message_isfile;
    const char *homedir;
    const char *docrootenv;
    int jailuser;
    int max_login_attempts;
    int limit_peruser;
    int limit_perip;
    int limit_perserver;
    const char *limitdbfile;
} ftp_server_config;

typedef struct ftp_dir_config
{
    const char *readme;
    const char *path;
    int readme_isfile;
    apr_fileperms_t fileperms;
    apr_fileperms_t dirperms;
} ftp_dir_config;

#define ftp_get_module_config(v) \
    ap_get_module_config(v, &ftp_module)
#define ftp_set_module_config(v,val) \
    ap_set_module_config(v, &ftp_module, val)

/* 
 * FTP connection structure.  Each connection to the server will
 * have an ftp_connection structure associated with it.
 */
typedef struct ftp_connection
{
    conn_rec *connection;     /* The connection */
    server_rec *orig_server;  /* Stow the original server for reset */
    int close_connection;     /* Final control command was processed */
    const char *response_notes; /* Notes for response handling. */

    /* User information */
    apr_pool_t *login_pool;   /* Child of c->pool reset on every USER cmd */
    int logged_in;
    const char *host;
    const char *user;
    const char *authorization;
    int login_attempts;

    /* Security information */
    int auth;                 /* The AUTH type.  One of NONE/SSL/TLS */
    int pbsz;                 /* Data protection buffer size.  Usually zero. */
    int prot;                 /* Data channel protection level. C/S/E/P */
    int is_secure;
    void *ssl_input_ctx;
    void *ssl_output_ctx;

    /* Connection stats */
    int files;                /* STOR/RETR transfer count */
    apr_off_t bytes;          /* Only STOR/RETR style traffic */
    int transfers;            /* All data transfers including LIST/NLST */
    apr_off_t traffic;        /* Total data channel traffic, plus HELP */

    /* Command state */
    char *cwd;                /* APR_PATH_MAX+1 buffer for Current Directory */
    char *rename_from;        /* APR_PATH_MAX+1 buffer for the RNFR command */
    int type;                 /* FTP transmission (TYPE_A/TYPE_I) */
    apr_off_t restart_point;  /* For data transfers */
    int filter_mask;          /* Filters required (CRLF, BYTERANGE, etc) */

    /* Data channel state */
    apr_pool_t *data_pool;    /* Child of c->pool reset on every data cmd */
    int all_epsv;             /* EPSV ALL command issued (refuse all others) */
    apr_socket_t *csock;      /* The local data socket we connect/accept on */
    apr_socket_t *datasock;   /* The remote data socket we send/recv on */
    apr_sockaddr_t *clientsa; /* The remote sa of this data connection */
    apr_time_t passive_created; /* Passive creation time or (-1) for active */

    /* Look-ahead for "next" command */
    apr_socket_t *cntlsock;   /* The control socket to watch for next cmd */
    apr_pool_t *next_pool;    /* Storage to get us to the next request_rec */
    apr_bucket_brigade *next_bb; /* Holding brigade for request look-ahead */
    char *next_request;       /* The next control command */
    apr_size_t next_reqsize;  /* Size of the next control command */
} ftp_connection;

/*
 * This is the generic registration function, 
 * see ftp_hook_cmd or ftp_hook_cmd_alias.
 */
typedef int ftp_hook_fn(request_rec *r, const char *arg);

FTP_DECLARE(void) ftp_hook_cmd_any(const char *key, ftp_hook_fn *pf, 
                                   const char *alias, int order, 
                                   int flags, const char *help);

/* Flags for each FTP command handler.
 *
 * FTP_TAKE0 - This command takes no arguments.
 * FTP_TAKE1 - This command takes a single argument.
 * FTP_KEEP_WHITESPACE  - leading and trailing spaces are significant.
 * FTP_NEED_LOGIN - The user needs to be logged in to execute this command.
 * FTP_DATA_INTR - The current data transfer is interrupted by this command.
 * FTP_NEW_FEAT - This command was introduced following RFC 2389, show in FEAT list.
 *                Also see comments for ftp_feat_advert() below.
 * FTP_NO_HELP - Implicit if the command key contains a space, this flag omits 
 *               a given FTP command from the HELP listing.
 * FTP_EXTENSIBLE - Upon all cmd handlers declining, reply is COMMAND_NOT_IMPL_PARAM
 *                  rather than COMMAND_NOT_IMPLEMENTED
 * FTP_TAKE1_PATH - This command takes a single argument, delimited by a single
 *                  space, with leading and trailing spaces significant.
 */
#define FTP_TAKE0            (1 << 0)
#define FTP_TAKE1            (1 << 1)
#define FTP_NEED_LOGIN       (1 << 5)
#define FTP_DATA_INTR        (1 << 6)
#define FTP_NEW_FEAT         (1 << 7)
#define FTP_NO_HELP          (1 << 8)
#define FTP_EXTENSIBLE       (1 << 9)
#define FTP_KEEP_WHITESPACE  (1 << 10)

#define FTP_TAKE1_PATH       (FTP_TAKE1 | FTP_KEEP_WHITESPACE)

/* FTP command handler ordering */
#define FTP_HOOK_FIRST  10
#define FTP_HOOK_MIDDLE 20
#define FTP_HOOK_LAST   30

/*
 * This this the main hook that is used to hook into mod_ftp.  The
 * key is the command, i.e. "DELE".  The ftp_hook_fn is the function 
 * pointer that will handle this request.  The flags variable is an 
 * integer that is used to tell the dispatcher (ftp_run_cmd) properties
 * about the handler that is about to be run.  These flags include 
 * whether the user needs to be logged in, and how many arguments we 
 * expect the command to have. See the flags above for details.  The 
 * final argument is the help string that will be displayed if the user
 * does "HELP cmd".
 */
#define ftp_hook_cmd(aliascmd, pf, order, flags, desc) \
    ftp_hook_cmd_any((aliascmd), (pf), NULL, (order), \
                     (flags), (desc))

/*
 * This this the alternate hook that is used to hook command aliases
 * into mod_ftp.  The key is the aliased command, i.e. "XPWD".  
 * The ftp_hook_fn is used instead to store the base command, i.e. "PWD".
 * The flags variable is an integer that is used to tell the dispatcher 
 * (ftp_run_cmd) properties about the handler that is about to be run.  
 * These flags must match the base command hook registration.  This macro
 * will always add FTP_ALIAS_CMD to the flags.  See the flags above for 
 * details of other flags.  The final argument is the help string that will 
 * be displayed if the user does "HELP cmd".
 */
#define ftp_hook_cmd_alias(aliascmd, basecmd, order, flags, desc) \
    ftp_hook_cmd_any((aliascmd), NULL, (basecmd), (order), \
                     (flags) , (desc))

/*
 * This this for registration of a non-command based feature which should
 * be reported by the FEAT query, but not the HELP query.  An example is
 * "REST STREAM", where the HELP command and hook is registered for REST,
 * but the FEAT advertised is "REST STREAM".
 */
#define ftp_feat_advert(feature) \
    ftp_hook_cmd_any((feature), NULL, NULL, FTP_HOOK_LAST, \
                     FTP_NEW_FEAT | FTP_NO_HELP, "")

/*
 * This command escapes text (not containing end of line) to protect the
 * the special characters CR and 0xFF, by trailing CR with IAC NOP (which
 * is effectively NULL) and 0xFF with a second 0xFF, for transmission over
 * the control connection as part of a single-line response.
 *
 * Note that the resulting string may be the origin string if transformation
 * is not required.
 */
FTP_DECLARE(const char *) ftp_escape_control_text(const char *s,
                                                  apr_pool_t *pool);

#ifdef __cplusplus
}
#endif

#endif

#endif

