/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.eventmesh.connector.canal.source;

import org.apache.eventmesh.common.config.connector.rdb.canal.CanalSourceIncrementConfig;
import org.apache.eventmesh.connector.canal.CanalConnectRecord;
import org.apache.eventmesh.connector.canal.model.EventColumn;
import org.apache.eventmesh.connector.canal.model.EventColumnIndexComparable;
import org.apache.eventmesh.connector.canal.model.EventType;
import org.apache.eventmesh.connector.canal.source.table.RdbTableMgr;

import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.springframework.util.CollectionUtils;

import com.alibaba.otter.canal.protocol.CanalEntry;
import com.alibaba.otter.canal.protocol.CanalEntry.Column;
import com.alibaba.otter.canal.protocol.CanalEntry.Entry;
import com.alibaba.otter.canal.protocol.CanalEntry.RowChange;
import com.alibaba.otter.canal.protocol.CanalEntry.RowData;

import lombok.extern.slf4j.Slf4j;

/**
 * data object parse
 */
@Slf4j
public class EntryParser {

    public static Map<Long, List<CanalConnectRecord>> parse(CanalSourceIncrementConfig sourceConfig, List<Entry> datas,
        RdbTableMgr tables) {
        List<CanalConnectRecord> recordList = new ArrayList<>();
        List<Entry> transactionDataBuffer = new ArrayList<>();
        // need check weather the entry is loopback
        boolean needSync;
        Map<Long, List<CanalConnectRecord>> recordMap = new HashMap<>();
        try {
            for (Entry entry : datas) {
                switch (entry.getEntryType()) {
                    case ROWDATA:
                        RowChange rowChange = RowChange.parseFrom(entry.getStoreValue());
                        // don't support gtid for mariadb
                        if (sourceConfig.getServerUUID() != null && sourceConfig.isGTIDMode() && !sourceConfig.isMariaDB()) {
                            if (checkGtidForEntry(entry, sourceConfig)) {
                                transactionDataBuffer.add(entry);
                            }
                        } else {
                            // if not gtid mode, need check weather the entry is loopback by specified column value
                            needSync = checkNeedSync(sourceConfig, rowChange);
                            if (needSync) {
                                log.debug("entry evenType {}|rowChange {}", rowChange.getEventType(), rowChange);
                                transactionDataBuffer.add(entry);
                            }
                        }
                        break;
                    case TRANSACTIONEND:
                        parseRecordListWithEntryBuffer(sourceConfig, recordList, transactionDataBuffer, tables);
                        if (!recordList.isEmpty()) {
                            List<CanalConnectRecord> transactionEndList = new ArrayList<>(recordList);
                            recordMap.put(entry.getHeader().getLogfileOffset(), transactionEndList);
                        }
                        recordList.clear();
                        transactionDataBuffer.clear();
                        break;
                    default:
                        break;
                }
            }

            // add last data in transactionDataBuffer, in case no TRANSACTIONEND
            parseRecordListWithEntryBuffer(sourceConfig, recordList, transactionDataBuffer, tables);
            if (!recordList.isEmpty()) {
                List<CanalConnectRecord> transactionEndList = new ArrayList<>(recordList);
                CanalConnectRecord lastCanalConnectRecord = transactionEndList.get(transactionEndList.size() - 1);
                recordMap.put(lastCanalConnectRecord.getBinLogOffset(), transactionEndList);
            }
            recordList.clear();
            transactionDataBuffer.clear();

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return recordMap;
    }

    private static boolean checkGtidForEntry(Entry entry, CanalSourceIncrementConfig sourceConfig) {
        String currentGtid = entry.getHeader().getPropsList().get(0).getValue();
        return currentGtid.contains(sourceConfig.getServerUUID());
    }

    private static void parseRecordListWithEntryBuffer(CanalSourceIncrementConfig sourceConfig,
        List<CanalConnectRecord> recordList,
        List<Entry> transactionDataBuffer, RdbTableMgr tables) {
        for (Entry bufferEntry : transactionDataBuffer) {
            List<CanalConnectRecord> recordParsedList = internParse(sourceConfig, bufferEntry, tables);
            if (CollectionUtils.isEmpty(recordParsedList)) {
                continue;
            }
            long totalSize = bufferEntry.getHeader().getEventLength();
            long eachSize = totalSize / recordParsedList.size();
            for (CanalConnectRecord record : recordParsedList) {
                if (record == null) {
                    continue;
                }
                record.setSize(eachSize);
                recordList.add(record);
            }
        }
    }

    private static boolean checkNeedSync(CanalSourceIncrementConfig sourceConfig, RowChange rowChange) {
        Column markedColumn = null;
        CanalEntry.EventType eventType = rowChange.getEventType();
        if (StringUtils.isEmpty(sourceConfig.getNeedSyncMarkTableColumnName())) {
            return true;
        }
        if (eventType.equals(CanalEntry.EventType.DELETE)) {
            markedColumn = getColumnIgnoreCase(rowChange.getRowDatas(0).getBeforeColumnsList(),
                sourceConfig.getNeedSyncMarkTableColumnName());
        } else if (eventType.equals(CanalEntry.EventType.INSERT) || eventType.equals(CanalEntry.EventType.UPDATE)) {
            markedColumn = getColumnIgnoreCase(rowChange.getRowDatas(0).getAfterColumnsList(),
                sourceConfig.getNeedSyncMarkTableColumnName());
        }
        if (markedColumn != null) {
            return StringUtils.equalsIgnoreCase(markedColumn.getValue(),
                sourceConfig.getNeedSyncMarkTableColumnValue());
        }
        return false;
    }

    private static Column getColumnIgnoreCase(List<Column> columns, String columName) {
        for (Column column : columns) {
            if (column.getName().equalsIgnoreCase(columName)) {
                return column;
            }
        }
        return null;
    }

    private static List<CanalConnectRecord> internParse(CanalSourceIncrementConfig sourceConfig, Entry entry,
        RdbTableMgr tableMgr) {
        String schemaName = entry.getHeader().getSchemaName();
        String tableName = entry.getHeader().getTableName();
        if (tableMgr.getTable(schemaName, tableName) == null) {
            return null;
        }

        RowChange rowChange = null;
        try {
            rowChange = RowChange.parseFrom(entry.getStoreValue());
        } catch (Exception e) {
            throw new RuntimeException("parser of canal-event has an error , data:" + entry.toString(), e);
        }

        if (rowChange == null) {
            return null;
        }

        EventType eventType = EventType.valueOf(rowChange.getEventType().name());

        if (eventType.isQuery()) {
            return null;
        }

        if (eventType.isDdl()) {
            log.warn("unsupported ddl event type: {}", eventType);
            return null;
        }

        List<CanalConnectRecord> recordList = new ArrayList<>();
        for (RowData rowData : rowChange.getRowDatasList()) {
            CanalConnectRecord record = internParse(sourceConfig, entry, rowChange, rowData);
            recordList.add(record);
        }

        return recordList;
    }

    private static CanalConnectRecord internParse(CanalSourceIncrementConfig canalSourceConfig, Entry entry,
        RowChange rowChange, RowData rowData) {
        CanalConnectRecord canalConnectRecord = new CanalConnectRecord();
        canalConnectRecord.setTableName(entry.getHeader().getTableName());
        canalConnectRecord.setSchemaName(entry.getHeader().getSchemaName());
        canalConnectRecord.setEventType(EventType.valueOf(rowChange.getEventType().name()));
        canalConnectRecord.setExecuteTime(entry.getHeader().getExecuteTime());
        canalConnectRecord.setJournalName(entry.getHeader().getLogfileName());
        canalConnectRecord.setBinLogOffset(entry.getHeader().getLogfileOffset());
        // if enabled gtid mode, gtid not null
        if (canalSourceConfig.isGTIDMode()) {
            if (canalSourceConfig.isMariaDB()) {
                String currentGtid = entry.getHeader().getGtid();
                canalConnectRecord.setGtid(currentGtid);
                canalConnectRecord.setCurrentGtid(currentGtid);
            } else {
                String currentGtid = entry.getHeader().getPropsList().get(0).getValue();
                String gtidRange = replaceGtidRange(entry.getHeader().getGtid(), currentGtid, canalSourceConfig.getServerUUID());
                canalConnectRecord.setGtid(gtidRange);
                canalConnectRecord.setCurrentGtid(currentGtid);
            }
        }

        EventType eventType = canalConnectRecord.getEventType();

        List<Column> beforeColumns = rowData.getBeforeColumnsList();
        List<Column> afterColumns = rowData.getAfterColumnsList();

        boolean isRowMode = canalSourceConfig.getSyncMode().isRow();

        Map<String, EventColumn> keyColumns = new LinkedHashMap<>();
        Map<String, EventColumn> oldKeyColumns = new LinkedHashMap<>();
        Map<String, EventColumn> notKeyColumns = new LinkedHashMap<>();

        if (eventType.isInsert()) {
            for (Column column : afterColumns) {
                if (column.getIsKey()) {
                    keyColumns.put(column.getName(), copyEventColumn(column, true));
                } else {
                    notKeyColumns.put(column.getName(), copyEventColumn(column, true));
                }
            }
        } else if (eventType.isDelete()) {
            for (Column column : beforeColumns) {
                if (column.getIsKey()) {
                    keyColumns.put(column.getName(), copyEventColumn(column, true));
                } else {
                    notKeyColumns.put(column.getName(), copyEventColumn(column, true));
                }
            }
        } else if (eventType.isUpdate()) {
            for (Column column : beforeColumns) {
                if (column.getIsKey()) {
                    oldKeyColumns.put(column.getName(), copyEventColumn(column, true));
                    keyColumns.put(column.getName(), copyEventColumn(column, true));
                } else {
                    if (isRowMode && entry.getHeader().getSourceType() == CanalEntry.Type.ORACLE) {
                        notKeyColumns.put(column.getName(), copyEventColumn(column, true));
                    }
                }
            }
            for (Column column : afterColumns) {
                if (column.getIsKey()) {
                    keyColumns.put(column.getName(), copyEventColumn(column, true));
                } else if (isRowMode || entry.getHeader().getSourceType() == CanalEntry.Type.ORACLE
                    || column.getUpdated()) {

                    boolean isUpdate = true;
                    if (entry.getHeader().getSourceType() == CanalEntry.Type.MYSQL) {
                        isUpdate = column.getUpdated();
                    }

                    notKeyColumns.put(column.getName(), copyEventColumn(column, isUpdate));
                }
            }

            if (entry.getHeader().getSourceType() == CanalEntry.Type.ORACLE) {
                checkUpdateKeyColumns(oldKeyColumns, keyColumns);
            }
        }

        List<EventColumn> keys = new ArrayList<>(keyColumns.values());
        List<EventColumn> oldKeys = new ArrayList<>(oldKeyColumns.values());
        List<EventColumn> columns = new ArrayList<>(notKeyColumns.values());

        keys.sort(new EventColumnIndexComparable());
        oldKeys.sort(new EventColumnIndexComparable());
        columns.sort(new EventColumnIndexComparable());
        if (!keyColumns.isEmpty()) {
            canalConnectRecord.setKeys(keys);
            if (canalConnectRecord.getEventType().isUpdate() && !oldKeys.equals(keys)) {
                canalConnectRecord.setOldKeys(oldKeys);
            }
            canalConnectRecord.setColumns(columns);
        } else {
            throw new RuntimeException("this row data has no pks , entry: " + entry + " and rowData: "
                + rowData);
        }

        return canalConnectRecord;
    }

    public static String replaceGtidRange(String gtid, String currentGtid, String serverUUID) {
        String[] gtidRangeArray = gtid.split(",");
        for (int i = 0; i < gtidRangeArray.length; i++) {
            String gtidRange = gtidRangeArray[i];
            if (gtidRange.startsWith(serverUUID)) {
                gtidRangeArray[i] = gtidRange.replaceFirst("\\d+$", currentGtid.split(":")[1]);
            }
        }
        return String.join(",", gtidRangeArray);
    }

    private static void checkUpdateKeyColumns(Map<String, EventColumn> oldKeyColumns,
        Map<String, EventColumn> keyColumns) {
        if (oldKeyColumns.isEmpty()) {
            return;
        }
        if (keyColumns.size() > oldKeyColumns.size()) {
            return;
        }

        if (keyColumns.isEmpty()) {
            keyColumns.putAll(oldKeyColumns);
            return;
        }

        if (oldKeyColumns.size() != keyColumns.size()) {
            for (String oldKey : oldKeyColumns.keySet()) {
                if (keyColumns.get(oldKey) == null) {
                    keyColumns.put(oldKey, oldKeyColumns.get(oldKey));
                }
            }
        }
    }

    private static EventColumn copyEventColumn(Column column, boolean isUpdate) {
        EventColumn eventColumn = new EventColumn();
        eventColumn.setIndex(column.getIndex());
        eventColumn.setKey(column.getIsKey());
        eventColumn.setNull(column.getIsNull());
        eventColumn.setColumnName(column.getName());
        eventColumn.setColumnValue(column.getValue());
        eventColumn.setUpdate(isUpdate);
        eventColumn.setColumnType(column.getSqlType());

        return eventColumn;
    }

}
