/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.druid.sql.calcite.external;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.calcite.plan.Convention;
import org.apache.calcite.plan.RelOptCluster;
import org.apache.calcite.rel.AbstractRelNode;
import org.apache.calcite.rel.RelWriter;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.druid.sql.calcite.table.DruidTable;
import org.apache.druid.sql.calcite.table.ExternalTable;

/**
 * Represents a scan of an external table. Generated by {@link ExternalTable},
 * which represents an {@link ExternalDataSource}.
 *
 * This class is exercised in CalciteInsertDmlTest but is not currently
 * exposed to end users.
 */
public class ExternalTableScan extends AbstractRelNode
{
  private final ObjectMapper jsonMapper;
  private final ExternalTable druidTable;

  public ExternalTableScan(
      final RelOptCluster cluster,
      final ObjectMapper jsonMapper,
      final ExternalTable druidTable
  )
  {
    super(cluster, cluster.traitSetOf(Convention.NONE));
    this.jsonMapper = jsonMapper;
    this.druidTable = druidTable;
  }

  public DruidTable getDruidTable()
  {
    return druidTable;
  }

  @Override
  protected RelDataType deriveRowType()
  {
    return druidTable.getRowType(getCluster().getTypeFactory());
  }

  @Override
  public RelWriter explainTerms(RelWriter pw)
  {
    final String dataSourceString;

    try {
      dataSourceString = jsonMapper.writeValueAsString(druidTable.getDataSource());
    }
    catch (JsonProcessingException e) {
      throw new RuntimeException(e);
    }

    return pw.item("dataSource", dataSourceString);
  }
}
