/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.storage.s3;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Supplier;
import com.google.common.collect.Lists;
import com.google.inject.Inject;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.MapUtils;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.segment.loading.DataSegmentKiller;
import org.apache.druid.segment.loading.SegmentLoadingException;
import org.apache.druid.storage.s3.S3DataSegmentPusherConfig;
import org.apache.druid.storage.s3.S3InputDataConfig;
import org.apache.druid.storage.s3.S3Utils;
import org.apache.druid.storage.s3.ServerSideEncryptingAmazonS3;
import org.apache.druid.timeline.DataSegment;

public class S3DataSegmentKiller
implements DataSegmentKiller {
    private static final Logger log = new Logger(S3DataSegmentKiller.class);
    private static final int MAX_MULTI_OBJECT_DELETE_SIZE = 1000;
    private static final String MULTI_OBJECT_DELETE_EXEPTION_ERROR_FORMAT = "message: [%s], code: [%s]";
    private final Supplier<ServerSideEncryptingAmazonS3> s3ClientSupplier;
    private final S3DataSegmentPusherConfig segmentPusherConfig;
    private final S3InputDataConfig inputDataConfig;

    @Inject
    public S3DataSegmentKiller(Supplier<ServerSideEncryptingAmazonS3> s3Client, S3DataSegmentPusherConfig segmentPusherConfig, S3InputDataConfig inputDataConfig) {
        this.s3ClientSupplier = s3Client;
        this.segmentPusherConfig = segmentPusherConfig;
        this.inputDataConfig = inputDataConfig;
    }

    public void kill(List<DataSegment> segments) throws SegmentLoadingException {
        if (segments.isEmpty()) {
            return;
        }
        if (segments.size() == 1) {
            this.kill(segments.get(0));
            return;
        }
        HashMap<String, List> bucketToKeysToDelete = new HashMap<String, List>();
        for (DataSegment segment : segments) {
            String s3Bucket = MapUtils.getString((Map)segment.getLoadSpec(), (String)"bucket");
            String path = MapUtils.getString((Map)segment.getLoadSpec(), (String)"key");
            List keysToDelete = bucketToKeysToDelete.computeIfAbsent(s3Bucket, k -> new ArrayList());
            keysToDelete.add(new DeleteObjectsRequest.KeyVersion(path));
            keysToDelete.add(new DeleteObjectsRequest.KeyVersion(DataSegmentKiller.descriptorPath((String)path)));
        }
        ServerSideEncryptingAmazonS3 s3Client = (ServerSideEncryptingAmazonS3)this.s3ClientSupplier.get();
        boolean shouldThrowException = false;
        for (Map.Entry bucketToKeys : bucketToKeysToDelete.entrySet()) {
            List keysToDelete;
            String s3Bucket = (String)bucketToKeys.getKey();
            boolean hadException = this.deleteKeysForBucket(s3Client, s3Bucket, keysToDelete = (List)bucketToKeys.getValue());
            if (!hadException) continue;
            shouldThrowException = true;
        }
        if (shouldThrowException) {
            throw new SegmentLoadingException("Couldn't delete segments from S3. See the task logs for more details.", new Object[0]);
        }
    }

    private boolean deleteKeysForBucket(ServerSideEncryptingAmazonS3 s3Client, String s3Bucket, List<DeleteObjectsRequest.KeyVersion> keysToDelete) {
        boolean hadException = false;
        DeleteObjectsRequest deleteObjectsRequest = new DeleteObjectsRequest(s3Bucket);
        deleteObjectsRequest.setQuiet(true);
        List keysChunks = Lists.partition(keysToDelete, (int)1000);
        for (List chunkOfKeys : keysChunks) {
            List keysToDeleteStrings = chunkOfKeys.stream().map(DeleteObjectsRequest.KeyVersion::getKey).collect(Collectors.toList());
            try {
                deleteObjectsRequest.setKeys(chunkOfKeys);
                log.info("Removing from bucket: [%s] the following index files: [%s] from s3!", new Object[]{s3Bucket, keysToDeleteStrings});
                S3Utils.retryS3Operation(() -> {
                    s3Client.deleteObjects(deleteObjectsRequest);
                    return null;
                }, 3);
            }
            catch (MultiObjectDeleteException e) {
                hadException = true;
                HashMap<String, List> errorToKeys = new HashMap<String, List>();
                for (MultiObjectDeleteException.DeleteError error : e.getErrors()) {
                    errorToKeys.computeIfAbsent(StringUtils.format((String)MULTI_OBJECT_DELETE_EXEPTION_ERROR_FORMAT, (Object[])new Object[]{error.getMessage(), error.getCode()}), k -> new ArrayList()).add(error.getKey());
                }
                errorToKeys.forEach((key, value) -> log.error("Unable to delete from bucket [%s], the following keys [%s], because [%s]", new Object[]{s3Bucket, String.join((CharSequence)", ", value), key}));
            }
            catch (AmazonServiceException e) {
                hadException = true;
                log.noStackTrace().warn((Throwable)e, "Unable to delete from bucket [%s], the following keys [%s]", new Object[]{s3Bucket, chunkOfKeys.stream().map(DeleteObjectsRequest.KeyVersion::getKey).collect(Collectors.joining(", "))});
            }
            catch (Exception e) {
                hadException = true;
                log.noStackTrace().warn((Throwable)e, "Unexpected exception occurred when deleting from bucket [%s], the following keys [%s]", new Object[]{s3Bucket, chunkOfKeys.stream().map(DeleteObjectsRequest.KeyVersion::getKey).collect(Collectors.joining(", "))});
            }
        }
        return hadException;
    }

    public void kill(DataSegment segment) throws SegmentLoadingException {
        try {
            Map loadSpec = segment.getLoadSpec();
            String s3Bucket = MapUtils.getString((Map)loadSpec, (String)"bucket");
            String s3Path = MapUtils.getString((Map)loadSpec, (String)"key");
            String s3DescriptorPath = DataSegmentKiller.descriptorPath((String)s3Path);
            ServerSideEncryptingAmazonS3 s3Client = (ServerSideEncryptingAmazonS3)this.s3ClientSupplier.get();
            if (s3Client.doesObjectExist(s3Bucket, s3Path)) {
                log.info("Removing index file[s3://%s/%s] from s3!", new Object[]{s3Bucket, s3Path});
                s3Client.deleteObject(s3Bucket, s3Path);
            }
            if (s3Client.doesObjectExist(s3Bucket, s3DescriptorPath)) {
                log.info("Removing descriptor file[s3://%s/%s] from s3!", new Object[]{s3Bucket, s3DescriptorPath});
                s3Client.deleteObject(s3Bucket, s3DescriptorPath);
            }
        }
        catch (AmazonServiceException e) {
            throw new SegmentLoadingException((Throwable)e, "Couldn't kill segment[%s]: [%s]", new Object[]{segment.getId(), e});
        }
    }

    public void killAll() throws IOException {
        if (this.segmentPusherConfig.getBucket() == null || this.segmentPusherConfig.getBaseKey() == null) {
            throw new ISE("Cannot delete all segment from S3 Deep Storage since druid.storage.bucket and druid.storage.baseKey are not both set.", new Object[0]);
        }
        log.info("Deleting all segment files from s3 location [bucket: '%s' prefix: '%s']", new Object[]{this.segmentPusherConfig.getBucket(), this.segmentPusherConfig.getBaseKey()});
        try {
            S3Utils.deleteObjectsInPath((ServerSideEncryptingAmazonS3)this.s3ClientSupplier.get(), this.inputDataConfig.getMaxListingLength(), this.segmentPusherConfig.getBucket(), this.segmentPusherConfig.getBaseKey(), (Predicate<S3ObjectSummary>)Predicates.alwaysTrue());
        }
        catch (Exception e) {
            log.error("Error occurred while deleting segment files from s3. Error: %s", new Object[]{e.getMessage()});
            throw new IOException(e);
        }
    }
}

