#!/usr/bin/env python
# encoding: utf-8

# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
# This file is copied from
# https://github.com/apache/impala/blob/branch-2.9.0/common/function-registry/impala_functions.py
# and modified by Doris

# This is a list of all the functions that are not auto-generated.
# It contains all the meta data that describes the function.

# The format is:
#   [sql aliases], <return_type>, [<args>], <nullable mode>, [template_types]
#
# 'sql aliases' are the function names that can be used from sql. There must be at least
# one per function.
#
# 'nullable mode' reflects whether the return value of the function is null. See @Function.NullableMode
# for the specific mode and meaning.
#
# 'template_types' is for template function just like C++. It is optional list.
# eg. [['element_at', '%element_extract%'], 'V', ['MAP<K, V>', 'K'], 'ALWAYS_NULLABLE', ['K', 'V']],
#     'K' and 'V' is type template and will be specialized at runtime in FE to match specific args.
#
# 'template_types' support variadic template is now support variadic template.
# eg. [['struct'], 'STRUCT<TYPES>', ['TYPES'], 'ALWAYS_NOT_NULLABLE', ['TYPES...']],
#     Inspired by C++ std::vector::emplace_back() function. 'TYPES...' is variadic template and will
#     be expanded to normal templates at runtime in FE to match variadic args. Please ensure that the
#     variadic template is placed at the last position of all templates.
visible_functions = {
    # Bit and Byte functions
    # For functions corresponding to builtin operators, we can reuse the implementations
    "bit" :[
        [['bitand'], 'TINYINT', ['TINYINT', 'TINYINT'], ''],
        [['bitand'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], ''],
        [['bitand'], 'INT', ['INT', 'INT'], ''],
        [['bitand'], 'BIGINT', ['BIGINT', 'BIGINT'], ''],
        [['bitand'], 'LARGEINT', ['LARGEINT', 'LARGEINT'], ''],
        
        [['bit_count'], 'TINYINT',   ['TINYINT'],     ''],
        [['bit_count'], 'TINYINT',   ['SMALLINT'],    ''],
        [['bit_count'], 'TINYINT',   ['INT'],         ''],
        [['bit_count'], 'TINYINT',   ['BIGINT'],      ''],
        [['bit_count'], 'SMALLINT',  ['LARGEINT'],    ''],

        [['bitor'], 'TINYINT', ['TINYINT', 'TINYINT'], ''],
        [['bitor'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], ''],
        [['bitor'], 'INT', ['INT', 'INT'], ''],
        [['bitor'], 'BIGINT', ['BIGINT', 'BIGINT'], ''],
        [['bitor'], 'LARGEINT', ['LARGEINT', 'LARGEINT'], ''],

        [['bitxor'], 'TINYINT', ['TINYINT', 'TINYINT'], ''],
        [['bitxor'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], ''],
        [['bitxor'], 'INT', ['INT', 'INT'], ''],
        [['bitxor'], 'BIGINT', ['BIGINT', 'BIGINT'], ''],
        [['bitxor'], 'LARGEINT', ['LARGEINT', 'LARGEINT'], ''],

        [['bitnot'], 'TINYINT', ['TINYINT'], ''],
        [['bitnot'], 'SMALLINT', ['SMALLINT'], ''],
        [['bitnot'], 'INT', ['INT'], ''],
        [['bitnot'], 'BIGINT', ['BIGINT'], ''],
        [['bitnot'], 'LARGEINT', ['LARGEINT'], ''],
        
        [['bit_shift_left'],    'BIGINT',   ['BIGINT',  'TINYINT'],     ''],
        [['bit_shift_right'],   'BIGINT',   ['BIGINT',  'TINYINT'],     ''],
        [['bit_test','bit_test_all'], 'TINYINT', ['TINYINT','TINYINT','...'], ''],
        [['bit_test','bit_test_all'], 'TINYINT', ['SMALLINT','SMALLINT','...'], ''],
        [['bit_test','bit_test_all'], 'TINYINT', ['INT','INT','...'], ''],
        [['bit_test','bit_test_all'], 'TINYINT', ['BIGINT','BIGINT','...'], ''],
        [['bit_test','bit_test_all'], 'TINYINT', ['LARGEINT','LARGEINT','...'], '']
    ],

    # map functions
    "map": [
        [['map'], 'MAP<K, V>', ['K', 'V', '...'], 'ALWAYS_NOT_NULLABLE', ['K', 'V']],
        [['element_at', '%element_extract%'], 'V', ['MAP<K, V>', 'K'], 'ALWAYS_NULLABLE', ['K', 'V']],
        [['size', 'map_size', 'cardinality'], 'BIGINT', ['MAP<K, V>'], '', ['K', 'V']],
        [['map_contains_key'], 'BOOLEAN', ['MAP<K, V>', 'K'], 'CUSTOM', ['K', 'V']],
        [['map_contains_value'], 'BOOLEAN', ['MAP<K, V>', 'V'], 'CUSTOM', ['K', 'V']],
        #[['map_contains_key_like'], 'BOOLEAN', ['MAP<K, V>', 'K'], '', ['K', 'V']],
        [['map_keys'], 'ARRAY<K>', ['MAP<K, V>'], '', ['K', 'V']],
        [['map_values'], 'ARRAY<V>', ['MAP<K, V>'], '', ['K', 'V']],
    ],

    # variant functions
    "variant": [
        [['element_at', '%element_extract%'], 'VARIANT', ['VARIANT', 'VARCHAR'], 'ALWAYS_NULLABLE'],
    ],

    # struct functions
    "struct": [
        [['struct'], 'STRUCT<TYPES>', ['TYPES'], 'ALWAYS_NOT_NULLABLE', ['TYPES...']],
        [['named_struct'], 'ANY_STRUCT_TYPE', ['TYPES'], 'ALWAYS_NOT_NULLABLE', ['TYPES...']],
        [['struct_element'], 'ANY_ELEMENT_TYPE', ['ANY_STRUCT_TYPE', 'INT'], 'ALWAYS_NULLABLE', ['TYPES...']],
        [['struct_element'], 'ANY_ELEMENT_TYPE', ['ANY_STRUCT_TYPE', 'VARCHAR'], 'ALWAYS_NULLABLE', ['TYPES...']],
    ],

    # array functions
    "array": [
        [['array'], 'ARRAY', ['BOOLEAN', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['TINYINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['SMALLINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['INT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['BIGINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['LARGEINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['DATETIME', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['DATE', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['DATETIMEV2', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['DATEV2', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['DECIMALV2', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['DECIMAL32', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['DECIMAL64', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['DECIMAL128', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['FLOAT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['DOUBLE', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['VARCHAR', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['array'], 'ARRAY', ['STRING', '...'], 'ALWAYS_NOT_NULLABLE'],

        [['element_at', '%element_extract%'], 'T', ['ARRAY<T>', 'BIGINT'], 'ALWAYS_NULLABLE', ['T']],

        [['l1_distance'], 'DOUBLE', ['ARRAY_TINYINT', 'ARRAY_TINYINT'], 'ALWAYS_NULLABLE'],
        [['l1_distance'], 'DOUBLE', ['ARRAY_SMALLINT', 'ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['l1_distance'], 'DOUBLE', ['ARRAY_INT', 'ARRAY_INT'], 'ALWAYS_NULLABLE'],
        [['l1_distance'], 'DOUBLE', ['ARRAY_BIGINT', 'ARRAY_BIGINT'], 'ALWAYS_NULLABLE'],
        [['l1_distance'], 'DOUBLE', ['ARRAY_LARGEINT', 'ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['l1_distance'], 'DOUBLE', ['ARRAY_FLOAT', 'ARRAY_FLOAT'], 'ALWAYS_NULLABLE'],
        [['l1_distance'], 'DOUBLE', ['ARRAY_DOUBLE', 'ARRAY_DOUBLE'], 'ALWAYS_NULLABLE'],

        [['l2_distance'], 'DOUBLE', ['ARRAY_TINYINT', 'ARRAY_TINYINT'], 'ALWAYS_NULLABLE'],
        [['l2_distance'], 'DOUBLE', ['ARRAY_SMALLINT', 'ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['l2_distance'], 'DOUBLE', ['ARRAY_INT', 'ARRAY_INT'], 'ALWAYS_NULLABLE'],
        [['l2_distance'], 'DOUBLE', ['ARRAY_BIGINT', 'ARRAY_BIGINT'], 'ALWAYS_NULLABLE'],
        [['l2_distance'], 'DOUBLE', ['ARRAY_LARGEINT', 'ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['l2_distance'], 'DOUBLE', ['ARRAY_FLOAT', 'ARRAY_FLOAT'], 'ALWAYS_NULLABLE'],
        [['l2_distance'], 'DOUBLE', ['ARRAY_DOUBLE', 'ARRAY_DOUBLE'], 'ALWAYS_NULLABLE'],

        [['cosine_distance'], 'DOUBLE', ['ARRAY_TINYINT', 'ARRAY_TINYINT'], 'ALWAYS_NULLABLE'],
        [['cosine_distance'], 'DOUBLE', ['ARRAY_SMALLINT', 'ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['cosine_distance'], 'DOUBLE', ['ARRAY_INT', 'ARRAY_INT'], 'ALWAYS_NULLABLE'],
        [['cosine_distance'], 'DOUBLE', ['ARRAY_BIGINT', 'ARRAY_BIGINT'], 'ALWAYS_NULLABLE'],
        [['cosine_distance'], 'DOUBLE', ['ARRAY_LARGEINT', 'ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['cosine_distance'], 'DOUBLE', ['ARRAY_FLOAT', 'ARRAY_FLOAT'], 'ALWAYS_NULLABLE'],
        [['cosine_distance'], 'DOUBLE', ['ARRAY_DOUBLE', 'ARRAY_DOUBLE'], 'ALWAYS_NULLABLE'],

        [['inner_product'], 'DOUBLE', ['ARRAY_TINYINT', 'ARRAY_TINYINT'], 'ALWAYS_NULLABLE'],
        [['inner_product'], 'DOUBLE', ['ARRAY_SMALLINT', 'ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['inner_product'], 'DOUBLE', ['ARRAY_INT', 'ARRAY_INT'], 'ALWAYS_NULLABLE'],
        [['inner_product'], 'DOUBLE', ['ARRAY_BIGINT', 'ARRAY_BIGINT'], 'ALWAYS_NULLABLE'],
        [['inner_product'], 'DOUBLE', ['ARRAY_LARGEINT', 'ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['inner_product'], 'DOUBLE', ['ARRAY_FLOAT', 'ARRAY_FLOAT'], 'ALWAYS_NULLABLE'],
        [['inner_product'], 'DOUBLE', ['ARRAY_DOUBLE', 'ARRAY_DOUBLE'], 'ALWAYS_NULLABLE'],

        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_BOOLEAN', 'ARRAY_BOOLEAN'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_TINYINT', 'ARRAY_TINYINT'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_SMALLINT', 'ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_INT', 'ARRAY_INT'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_BIGINT', 'ARRAY_BIGINT'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_LARGEINT', 'ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_DATETIME', 'ARRAY_DATETIME'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_DATE', 'ARRAY_DATE'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_DATETIMEV2', 'ARRAY_DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_DATEV2', 'ARRAY_DATEV2'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_FLOAT', 'ARRAY_FLOAT'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_DOUBLE', 'ARRAY_DOUBLE'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_DECIMALV2', 'ARRAY_DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_DECIMAL32', 'ARRAY_DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_DECIMAL64', 'ARRAY_DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_DECIMAL128', 'ARRAY_DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_VARCHAR', 'ARRAY_VARCHAR'], 'ALWAYS_NULLABLE'],
        [['arrays_overlap'], 'BOOLEAN', ['ARRAY_STRING', 'ARRAY_STRING'], 'ALWAYS_NULLABLE'],
        [['array_contains_all'], 'BOOLEAN', ['ARRAY<T>', 'ARRAY<T>'], '', ['T']],

        [['array_contains'], 'BOOLEAN', ['ARRAY<T>', 'T'], 'CUSTOM', ['T']],

        [['array_cum_sum'], 'ARRAY_BIGINT',     ['ARRAY_TINYINT'], ''],
        [['array_cum_sum'], 'ARRAY_BIGINT',     ['ARRAY_SMALLINT'], ''],
        [['array_cum_sum'], 'ARRAY_BIGINT',     ['ARRAY_INT'], ''],
        [['array_cum_sum'], 'ARRAY_BIGINT',     ['ARRAY_BIGINT'], ''],
        [['array_cum_sum'], 'ARRAY_LARGEINT',   ['ARRAY_LARGEINT'], ''],
        [['array_cum_sum'], 'ARRAY_DOUBLE',     ['ARRAY_FLOAT'], ''],
        [['array_cum_sum'], 'ARRAY_DOUBLE',     ['ARRAY_DOUBLE'], ''],
        [['array_cum_sum'], 'ARRAY_DECIMALV2',  ['ARRAY_DECIMALV2'], ''],
        [['array_cum_sum'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL32'], ''],
        [['array_cum_sum'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL64'], ''],
        [['array_cum_sum'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],

        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_BOOLEAN'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_TINYINT'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_SMALLINT'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_INT'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_BIGINT'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_LARGEINT'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_DATETIME'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_DATE'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_DATETIMEV2'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_DATEV2'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_FLOAT'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_DOUBLE'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_DECIMALV2'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_DECIMAL32'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_DECIMAL64'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_DECIMAL128'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_VARCHAR'], ''],
        [['array_enumerate'], 'ARRAY_BIGINT', ['ARRAY_STRING'], ''],

        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_BOOLEAN', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_TINYINT', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_SMALLINT', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_INT', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_BIGINT', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_LARGEINT', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_DATETIME', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_DATE', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_DATETIMEV2', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_DATEV2', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_FLOAT', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_DOUBLE', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_DECIMALV2', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_DECIMAL32', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_DECIMAL64', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_DECIMAL128', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_VARCHAR', '...'], ''],
        [['array_enumerate_uniq'], 'ARRAY_BIGINT', ['ARRAY_STRING', '...'], ''],

        [['countequal'], 'BIGINT', ['ARRAY_BOOLEAN', 'BOOLEAN'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_TINYINT', 'TINYINT'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_SMALLINT', 'SMALLINT'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_INT', 'INT'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_BIGINT', 'BIGINT'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_LARGEINT', 'LARGEINT'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_DATETIME', 'DATETIME'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_DATE', 'DATE'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_DATETIMEV2', 'DATETIMEV2'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_DATEV2', 'DATEV2'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_FLOAT', 'FLOAT'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_DOUBLE', 'DOUBLE'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_DECIMALV2', 'DECIMALV2'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_DECIMAL32', 'DECIMAL32'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_DECIMAL64', 'DECIMAL64'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_DECIMAL128', 'DECIMAL128'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_VARCHAR', 'VARCHAR'], 'CUSTOM'],
        [['countequal'], 'BIGINT', ['ARRAY_STRING', 'STRING'], 'CUSTOM'],

        [['array_position'], 'BIGINT', ['ARRAY_BOOLEAN', 'BOOLEAN'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_TINYINT', 'TINYINT'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_SMALLINT', 'SMALLINT'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_INT', 'INT'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_BIGINT', 'BIGINT'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_LARGEINT', 'LARGEINT'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_DATETIME', 'DATETIME'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_DATE', 'DATE'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_DATETIMEV2', 'DATETIMEV2'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_DATEV2', 'DATEV2'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_FLOAT', 'FLOAT'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_DOUBLE', 'DOUBLE'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_DECIMALV2', 'DECIMALV2'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_DECIMAL32', 'DECIMAL32'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_DECIMAL64', 'DECIMAL64'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_DECIMAL128', 'DECIMAL128'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_VARCHAR', 'VARCHAR'], 'CUSTOM'],
        [['array_position'], 'BIGINT', ['ARRAY_STRING', 'STRING'], 'CUSTOM'],

        [['cardinality', 'size', 'array_size'], 'BIGINT', ['ARRAY<T>'], '', ['T']],
        [['array_distinct'], 'ARRAY_BOOLEAN',   ['ARRAY_BOOLEAN'], ''],
        [['array_distinct'], 'ARRAY_TINYINT',   ['ARRAY_TINYINT'], ''],
        [['array_distinct'], 'ARRAY_SMALLINT',  ['ARRAY_SMALLINT'], ''],
        [['array_distinct'], 'ARRAY_INT',       ['ARRAY_INT'], ''],
        [['array_distinct'], 'ARRAY_BIGINT',    ['ARRAY_BIGINT'], ''],
        [['array_distinct'], 'ARRAY_LARGEINT',  ['ARRAY_LARGEINT'], ''],
        [['array_distinct'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME'], ''],
        [['array_distinct'], 'ARRAY_DATE',      ['ARRAY_DATE'], ''],
        [['array_distinct'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2'], ''],
        [['array_distinct'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2'], ''],
        [['array_distinct'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT'], ''],
        [['array_distinct'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE'], ''],
        [['array_distinct'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], ''],
        [['array_distinct'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], ''],
        [['array_distinct'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], ''],
        [['array_distinct'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],
        [['array_distinct'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR'], ''],
        [['array_distinct'], 'ARRAY_STRING',    ['ARRAY_STRING'], ''],

        [['array_difference'], 'ARRAY_SMALLINT',  ['ARRAY_TINYINT'], ''],
        [['array_difference'], 'ARRAY_INT',       ['ARRAY_SMALLINT'], ''],
        [['array_difference'], 'ARRAY_BIGINT',    ['ARRAY_INT'], ''],
        [['array_difference'], 'ARRAY_LARGEINT',  ['ARRAY_BIGINT'], ''],
        [['array_difference'], 'ARRAY_LARGEINT',  ['ARRAY_LARGEINT'], ''],
        [['array_difference'], 'ARRAY_DOUBLE',    ['ARRAY_FLOAT'], ''],
        [['array_difference'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE'], ''],
        [['array_difference'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], ''],
        [['array_difference'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], ''],
        [['array_difference'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], ''],
        [['array_difference'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],

        [['array_sort'], 'ARRAY_BOOLEAN',   ['ARRAY_BOOLEAN'], ''],
        [['array_sort'], 'ARRAY_TINYINT',   ['ARRAY_TINYINT'], ''],
        [['array_sort'], 'ARRAY_SMALLINT',  ['ARRAY_SMALLINT'], ''],
        [['array_sort'], 'ARRAY_INT',       ['ARRAY_INT'], ''],
        [['array_sort'], 'ARRAY_BIGINT',    ['ARRAY_BIGINT'], ''],
        [['array_sort'], 'ARRAY_LARGEINT',  ['ARRAY_LARGEINT'], ''],
        [['array_sort'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME'], ''],
        [['array_sort'], 'ARRAY_DATE',      ['ARRAY_DATE'], ''],
        [['array_sort'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2'], ''],
        [['array_sort'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2'], ''],
        [['array_sort'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT'], ''],
        [['array_sort'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE'], ''],
        [['array_sort'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], ''],
        [['array_sort'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], ''],
        [['array_sort'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], ''],
        [['array_sort'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],
        [['array_sort'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR'], ''],
        [['array_sort'], 'ARRAY_STRING',    ['ARRAY_STRING'], ''],

        [['array_reverse_sort'], 'ARRAY_BOOLEAN',   ['ARRAY_BOOLEAN'], ''],
        [['array_reverse_sort'], 'ARRAY_TINYINT',   ['ARRAY_TINYINT'], ''],
        [['array_reverse_sort'], 'ARRAY_SMALLINT',  ['ARRAY_SMALLINT'], ''],
        [['array_reverse_sort'], 'ARRAY_INT',       ['ARRAY_INT'], ''],
        [['array_reverse_sort'], 'ARRAY_BIGINT',    ['ARRAY_BIGINT'], ''],
        [['array_reverse_sort'], 'ARRAY_LARGEINT',  ['ARRAY_LARGEINT'], ''],
        [['array_reverse_sort'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME'], ''],
        [['array_reverse_sort'], 'ARRAY_DATE',      ['ARRAY_DATE'], ''],
        [['array_reverse_sort'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2'], ''],
        [['array_reverse_sort'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2'], ''],
        [['array_reverse_sort'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT'], ''],
        [['array_reverse_sort'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE'], ''],
        [['array_reverse_sort'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], ''],
        [['array_reverse_sort'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], ''],
        [['array_reverse_sort'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], ''],
        [['array_reverse_sort'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],
        [['array_reverse_sort'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR'], ''],
        [['array_reverse_sort'], 'ARRAY_STRING',    ['ARRAY_STRING'], ''],

        # array_join takes two params
        [['array_join'], 'STRING',   ['ARRAY_BOOLEAN','VARCHAR'], ''],
        [['array_join'], 'STRING',   ['ARRAY_TINYINT','VARCHAR'], ''],
        [['array_join'], 'STRING',  ['ARRAY_SMALLINT','VARCHAR'], ''],
        [['array_join'], 'STRING',       ['ARRAY_INT','VARCHAR'], ''],
        [['array_join'], 'STRING',    ['ARRAY_BIGINT','VARCHAR'], ''],
        [['array_join'], 'STRING',  ['ARRAY_LARGEINT','VARCHAR'], ''],
        [['array_join'], 'STRING',  ['ARRAY_DATETIME','VARCHAR'], ''],
        [['array_join'], 'STRING',      ['ARRAY_DATE','VARCHAR'], ''],
        [['array_join'], 'STRING',  ['ARRAY_DATETIMEV2','VARCHAR'], ''],
        [['array_join'], 'STRING',      ['ARRAY_DATEV2','VARCHAR'], ''],
        [['array_join'], 'STRING',     ['ARRAY_FLOAT','VARCHAR'], ''],
        [['array_join'], 'STRING',    ['ARRAY_DOUBLE','VARCHAR'], ''],
        [['array_join'], 'STRING', ['ARRAY_DECIMALV2','VARCHAR'], ''],
        [['array_join'], 'STRING', ['ARRAY_DECIMAL32','VARCHAR'], ''],
        [['array_join'], 'STRING', ['ARRAY_DECIMAL64','VARCHAR'], ''],
        [['array_join'], 'STRING', ['ARRAY_DECIMAL128','VARCHAR'], ''],
        [['array_join'], 'STRING',   ['ARRAY_VARCHAR','VARCHAR'], ''],
        [['array_join'], 'STRING',    ['ARRAY_STRING','VARCHAR'], ''],
        # array_join takes three params
        [['array_join'], 'STRING',   ['ARRAY_BOOLEAN','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',   ['ARRAY_TINYINT','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',  ['ARRAY_SMALLINT','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',       ['ARRAY_INT','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',    ['ARRAY_BIGINT','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',  ['ARRAY_LARGEINT','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',  ['ARRAY_DATETIME','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',      ['ARRAY_DATE','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',  ['ARRAY_DATETIMEV2','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',      ['ARRAY_DATEV2','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',     ['ARRAY_FLOAT','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',    ['ARRAY_DOUBLE','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING', ['ARRAY_DECIMALV2','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING', ['ARRAY_DECIMAL32','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING', ['ARRAY_DECIMAL64','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING', ['ARRAY_DECIMAL128','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',   ['ARRAY_VARCHAR','VARCHAR', 'VARCHAR'], ''],
        [['array_join'], 'STRING',    ['ARRAY_STRING','VARCHAR', 'VARCHAR'], ''],

        [['array_min'],     'BOOLEAN',  ['ARRAY_BOOLEAN'],  'ALWAYS_NULLABLE'],
        [['array_min'],     'TINYINT',  ['ARRAY_TINYINT'],  'ALWAYS_NULLABLE'],
        [['array_min'],     'SMALLINT', ['ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['array_min'],     'INT',      ['ARRAY_INT'],      'ALWAYS_NULLABLE'],
        [['array_min'],     'BIGINT',   ['ARRAY_BIGINT'],   'ALWAYS_NULLABLE'],
        [['array_min'],     'LARGEINT', ['ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['array_min'],     'FLOAT',    ['ARRAY_FLOAT'],    'ALWAYS_NULLABLE'],
        [['array_min'],     'DOUBLE',   ['ARRAY_DOUBLE'],   'ALWAYS_NULLABLE'],
        [['array_min'],     'DECIMALV2',['ARRAY_DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['array_min'],     'DECIMAL32',['ARRAY_DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['array_min'],     'DECIMAL64',['ARRAY_DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['array_min'],     'DECIMAL128',['ARRAY_DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['array_min'],     'DATE',     ['ARRAY_DATE'],     'ALWAYS_NULLABLE'],
        [['array_min'],     'DATETIME', ['ARRAY_DATETIME'], 'ALWAYS_NULLABLE'],
        [['array_min'],     'DATEV2',     ['ARRAY_DATEV2'],     'ALWAYS_NULLABLE'],
        [['array_min'],     'DATETIMEV2', ['ARRAY_DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['array_max'],     'BOOLEAN',  ['ARRAY_BOOLEAN'],  'ALWAYS_NULLABLE'],
        [['array_max'],     'TINYINT',  ['ARRAY_TINYINT'],  'ALWAYS_NULLABLE'],
        [['array_max'],     'SMALLINT', ['ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['array_max'],     'INT',      ['ARRAY_INT'],      'ALWAYS_NULLABLE'],
        [['array_max'],     'BIGINT',   ['ARRAY_BIGINT'],   'ALWAYS_NULLABLE'],
        [['array_max'],     'LARGEINT', ['ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['array_max'],     'FLOAT',    ['ARRAY_FLOAT'],    'ALWAYS_NULLABLE'],
        [['array_max'],     'DOUBLE',   ['ARRAY_DOUBLE'],   'ALWAYS_NULLABLE'],
        [['array_max'],     'DECIMALV2',['ARRAY_DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['array_max'],     'DECIMAL32',['ARRAY_DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['array_max'],     'DECIMAL64',['ARRAY_DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['array_max'],     'DECIMAL128',['ARRAY_DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['array_max'],     'DATE',     ['ARRAY_DATE'],     'ALWAYS_NULLABLE'],
        [['array_max'],     'DATETIME', ['ARRAY_DATETIME'], 'ALWAYS_NULLABLE'],
        [['array_max'],     'DATEV2',     ['ARRAY_DATEV2'],     'ALWAYS_NULLABLE'],
        [['array_max'],     'DATETIMEV2', ['ARRAY_DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['array_sum'],     'BIGINT',   ['ARRAY_BOOLEAN'],  'ALWAYS_NULLABLE'],
        [['array_sum'],     'BIGINT',   ['ARRAY_TINYINT'],  'ALWAYS_NULLABLE'],
        [['array_sum'],     'BIGINT',   ['ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['array_sum'],     'BIGINT',   ['ARRAY_INT'],      'ALWAYS_NULLABLE'],
        [['array_sum'],     'BIGINT',   ['ARRAY_BIGINT'],   'ALWAYS_NULLABLE'],
        [['array_sum'],     'LARGEINT', ['ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['array_sum'],     'DOUBLE',   ['ARRAY_FLOAT'], 'ALWAYS_NULLABLE'],
        [['array_sum'],     'DOUBLE',   ['ARRAY_DOUBLE'],   'ALWAYS_NULLABLE'],
        [['array_sum'],     'DECIMALV2',['ARRAY_DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['array_sum'],     'DECIMAL128',['ARRAY_DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['array_sum'],     'DECIMAL128',['ARRAY_DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['array_sum'],     'DECIMAL128',['ARRAY_DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['array_avg'],     'DOUBLE',   ['ARRAY_BOOLEAN'],  'ALWAYS_NULLABLE'],
        [['array_avg'],     'DOUBLE',   ['ARRAY_TINYINT'],  'ALWAYS_NULLABLE'],
        [['array_avg'],     'DOUBLE',   ['ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['array_avg'],     'DOUBLE',   ['ARRAY_INT'],      'ALWAYS_NULLABLE'],
        [['array_avg'],     'DOUBLE',   ['ARRAY_BIGINT'],   'ALWAYS_NULLABLE'],
        [['array_avg'],     'DOUBLE',   ['ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['array_avg'],     'DOUBLE',   ['ARRAY_FLOAT'],    'ALWAYS_NULLABLE'],
        [['array_avg'],     'DOUBLE',   ['ARRAY_DOUBLE'],   'ALWAYS_NULLABLE'],
        [['array_avg'],     'DECIMALV2',['ARRAY_DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['array_avg'],     'DECIMAL128',['ARRAY_DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['array_avg'],     'DECIMAL128',['ARRAY_DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['array_avg'],     'DECIMAL128',['ARRAY_DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['array_product'], 'DOUBLE',   ['ARRAY_BOOLEAN'],  'ALWAYS_NULLABLE'],
        [['array_product'], 'DOUBLE',   ['ARRAY_TINYINT'],  'ALWAYS_NULLABLE'],
        [['array_product'], 'DOUBLE',   ['ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['array_product'], 'DOUBLE',   ['ARRAY_INT'],      'ALWAYS_NULLABLE'],
        [['array_product'], 'DOUBLE',   ['ARRAY_BIGINT'],   'ALWAYS_NULLABLE'],
        [['array_product'], 'DOUBLE',   ['ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['array_product'], 'DOUBLE',   ['ARRAY_FLOAT'],    'ALWAYS_NULLABLE'],
        [['array_product'], 'DOUBLE',   ['ARRAY_DOUBLE'],   'ALWAYS_NULLABLE'],
        [['array_product'], 'DECIMALV2',['ARRAY_DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['array_product'], 'DECIMAL128',['ARRAY_DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['array_product'], 'DECIMAL128',['ARRAY_DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['array_product'], 'DECIMAL128',['ARRAY_DECIMAL128'], 'ALWAYS_NULLABLE'],

        [['array_remove'], 'ARRAY_BOOLEAN',   ['ARRAY_BOOLEAN', 'BOOLEAN'], ''],
        [['array_remove'], 'ARRAY_TINYINT',   ['ARRAY_TINYINT', 'TINYINT'], ''],
        [['array_remove'], 'ARRAY_SMALLINT',  ['ARRAY_SMALLINT', 'SMALLINT'], ''],
        [['array_remove'], 'ARRAY_INT',       ['ARRAY_INT', 'INT'], ''],
        [['array_remove'], 'ARRAY_BIGINT',    ['ARRAY_BIGINT', 'BIGINT'], ''],
        [['array_remove'], 'ARRAY_LARGEINT',  ['ARRAY_LARGEINT', 'LARGEINT'], ''],
        [['array_remove'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT', 'FLOAT'], ''],
        [['array_remove'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE', 'DOUBLE'], ''],
        [['array_remove'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2', 'DECIMALV2'], ''],
        [['array_remove'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32', 'DECIMAL32'], ''],
        [['array_remove'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64', 'DECIMAL64'], ''],
        [['array_remove'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'DECIMAL128'], ''],
        [['array_remove'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME', 'DATETIME'], ''],
        [['array_remove'], 'ARRAY_DATE',      ['ARRAY_DATE', 'DATE'], ''],
        [['array_remove'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2', 'DATETIMEV2'], ''],
        [['array_remove'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2', 'DATEV2'], ''],
        [['array_remove'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR', 'VARCHAR'], ''],
        [['array_remove'], 'ARRAY_STRING',    ['ARRAY_STRING', 'STRING'], ''],

        [['array_union'], 'ARRAY_BOOLEAN',  ['ARRAY_BOOLEAN', 'ARRAY_BOOLEAN', '...'],  ''],
        [['array_union'], 'ARRAY_TINYINT',  ['ARRAY_TINYINT', 'ARRAY_TINYINT', '...'],  ''],
        [['array_union'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT', 'ARRAY_SMALLINT', '...'], ''],
        [['array_union'], 'ARRAY_INT',      ['ARRAY_INT', 'ARRAY_INT', '...'], ''],
        [['array_union'], 'ARRAY_BIGINT',   ['ARRAY_BIGINT', 'ARRAY_BIGINT', '...'],   ''],
        [['array_union'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT', 'ARRAY_LARGEINT', '...'], ''],
        [['array_union'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT', 'ARRAY_FLOAT', '...'],  ''],
        [['array_union'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE', 'ARRAY_DOUBLE', '...'], ''],
        [['array_union'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2', 'ARRAY_DECIMALV2', '...'], ''],
        [['array_union'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32', 'ARRAY_DECIMAL32', '...'], ''],
        [['array_union'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64', 'ARRAY_DECIMAL64', '...'], ''],
        [['array_union'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'ARRAY_DECIMAL128', '...'], ''],
        [['array_union'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME', 'ARRAY_DATETIME', '...'], ''],
        [['array_union'], 'ARRAY_DATE',      ['ARRAY_DATE', 'ARRAY_DATE', '...'], ''],
        [['array_union'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2', 'ARRAY_DATETIMEV2', '...'], ''],
        [['array_union'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2', 'ARRAY_DATEV2', '...'], ''],
        [['array_union'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR', 'ARRAY_VARCHAR', '...'], ''],
        [['array_union'], 'ARRAY_STRING',    ['ARRAY_STRING', 'ARRAY_STRING', '...'], ''],

        [['array_apply'], 'ARRAY_BOOLEAN',  ['ARRAY_BOOLEAN', 'VARCHAR', 'BOOLEAN'],  ''],
        [['array_apply'], 'ARRAY_TINYINT',  ['ARRAY_TINYINT', 'VARCHAR', 'TINYINT'],  ''],
        [['array_apply'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT', 'VARCHAR', 'SMALLINT'], ''],
        [['array_apply'], 'ARRAY_INT',      ['ARRAY_INT', 'VARCHAR', 'INT'], ''],
        [['array_apply'], 'ARRAY_BIGINT',   ['ARRAY_BIGINT', 'VARCHAR', 'BIGINT'],   ''],
        [['array_apply'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT', 'VARCHAR', 'LARGEINT'], ''],
        [['array_apply'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT', 'VARCHAR', 'FLOAT'],  ''],
        [['array_apply'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE', 'VARCHAR', 'DOUBLE'], ''],
        [['array_apply'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2', 'VARCHAR', 'DECIMALV2'], ''],
        [['array_apply'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32', 'VARCHAR', 'DECIMAL32'], ''],
        [['array_apply'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64', 'VARCHAR', 'DECIMAL64'], ''],
        [['array_apply'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'VARCHAR', 'DECIMAL128'], ''],
        [['array_apply'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME', 'VARCHAR', 'DATETIME'], ''],
        [['array_apply'], 'ARRAY_DATE',      ['ARRAY_DATE', 'VARCHAR', 'DATE'], ''],
        [['array_apply'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2', 'VARCHAR', 'DATETIMEV2'], ''],
        [['array_apply'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2', 'VARCHAR', 'DATEV2'], ''],

        [['array_concat'], 'ARRAY_BOOLEAN',  ['ARRAY_BOOLEAN', '...'],  ''],
        [['array_concat'], 'ARRAY_TINYINT',  ['ARRAY_TINYINT', '...'],  ''],
        [['array_concat'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT', '...'], ''],
        [['array_concat'], 'ARRAY_INT',      ['ARRAY_INT', '...'], ''],
        [['array_concat'], 'ARRAY_BIGINT',   ['ARRAY_BIGINT', '...'],   ''],
        [['array_concat'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT', '...'], ''],
        [['array_concat'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT', '...'],  ''],
        [['array_concat'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE', '...'], ''],
        [['array_concat'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2', '...'], ''],
        [['array_concat'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32', '...'], ''],
        [['array_concat'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64', '...'], ''],
        [['array_concat'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', '...'], ''],
        [['array_concat'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME', '...'], ''],
        [['array_concat'], 'ARRAY_DATE',      ['ARRAY_DATE', '...'], ''],
        [['array_concat'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2', '...'], ''],
        [['array_concat'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2', '...'], ''],
        [['array_concat'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR', '...'], ''],
        [['array_concat'], 'ARRAY_STRING',    ['ARRAY_STRING', '...'], ''],

        [['array_except'], 'ARRAY_BOOLEAN',  ['ARRAY_BOOLEAN', 'ARRAY_BOOLEAN'],  ''],
        [['array_except'], 'ARRAY_TINYINT',  ['ARRAY_TINYINT', 'ARRAY_TINYINT'],  ''],
        [['array_except'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT', 'ARRAY_SMALLINT'], ''],
        [['array_except'], 'ARRAY_INT',      ['ARRAY_INT', 'ARRAY_INT'], ''],
        [['array_except'], 'ARRAY_BIGINT',   ['ARRAY_BIGINT', 'ARRAY_BIGINT'],   ''],
        [['array_except'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT', 'ARRAY_LARGEINT'], ''],
        [['array_except'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT', 'ARRAY_FLOAT'],  ''],
        [['array_except'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE', 'ARRAY_DOUBLE'], ''],
        [['array_except'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2', 'ARRAY_DECIMALV2'], ''],
        [['array_except'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32', 'ARRAY_DECIMAL32'], ''],
        [['array_except'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64', 'ARRAY_DECIMAL64'], ''],
        [['array_except'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'ARRAY_DECIMAL128'], ''],
        [['array_except'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME', 'ARRAY_DATETIME'], ''],
        [['array_except'], 'ARRAY_DATE',      ['ARRAY_DATE', 'ARRAY_DATE'], ''],
        [['array_except'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2', 'ARRAY_DATETIMEV2'], ''],
        [['array_except'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2', 'ARRAY_DATEV2'], ''],
        [['array_except'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR', 'ARRAY_VARCHAR'], ''],
        [['array_except'], 'ARRAY_STRING',    ['ARRAY_STRING', 'ARRAY_STRING'], ''],

        [['array_compact'], 'ARRAY_BOOLEAN',   ['ARRAY_BOOLEAN'], ''],
        [['array_compact'], 'ARRAY_TINYINT',   ['ARRAY_TINYINT'], ''],
        [['array_compact'], 'ARRAY_SMALLINT',  ['ARRAY_SMALLINT'], ''],
        [['array_compact'], 'ARRAY_INT',       ['ARRAY_INT'], ''],
        [['array_compact'], 'ARRAY_BIGINT',    ['ARRAY_BIGINT'], ''],
        [['array_compact'], 'ARRAY_LARGEINT',  ['ARRAY_LARGEINT'], ''],
        [['array_compact'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME'], ''],
        [['array_compact'], 'ARRAY_DATE',      ['ARRAY_DATE'], ''],
        [['array_compact'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2'], ''],
        [['array_compact'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2'], ''],
        [['array_compact'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT'], ''],
        [['array_compact'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE'], ''],
        [['array_compact'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], ''],
        [['array_compact'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], ''],
        [['array_compact'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], ''],
        [['array_compact'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],
        [['array_compact'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR'], ''],

        [['array_intersect'], 'ARRAY_BOOLEAN',  ['ARRAY_BOOLEAN', 'ARRAY_BOOLEAN', '...'],  ''],
        [['array_intersect'], 'ARRAY_TINYINT',  ['ARRAY_TINYINT', 'ARRAY_TINYINT', '...'],  ''],
        [['array_intersect'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT', 'ARRAY_SMALLINT', '...'], ''],
        [['array_intersect'], 'ARRAY_INT',      ['ARRAY_INT', 'ARRAY_INT', '...'], ''],
        [['array_intersect'], 'ARRAY_BIGINT',   ['ARRAY_BIGINT', 'ARRAY_BIGINT', '...'],   ''],
        [['array_intersect'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT', 'ARRAY_LARGEINT', '...'], ''],
        [['array_intersect'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT', 'ARRAY_FLOAT', '...'],  ''],
        [['array_intersect'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE', 'ARRAY_DOUBLE', '...'], ''],
        [['array_intersect'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2', 'ARRAY_DECIMALV2', '...'], ''],
        [['array_intersect'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32', 'ARRAY_DECIMAL32', '...'], ''],
        [['array_intersect'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64', 'ARRAY_DECIMAL64', '...'], ''],
        [['array_intersect'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'ARRAY_DECIMAL128', '...'], ''],
        [['array_intersect'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME', 'ARRAY_DATETIME', '...'], ''],
        [['array_intersect'], 'ARRAY_DATE',      ['ARRAY_DATE', 'ARRAY_DATE', '...'], ''],
        [['array_intersect'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2', 'ARRAY_DATETIMEV2', '...'], ''],
        [['array_intersect'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2', 'ARRAY_DATEV2', '...'], ''],
        [['array_intersect'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR', 'ARRAY_VARCHAR', '...'], ''],
        [['array_intersect'], 'ARRAY_STRING',    ['ARRAY_STRING', 'ARRAY_STRING', '...'], ''],

        [['array_slice', '%element_slice%'], 'ARRAY_BOOLEAN', ['ARRAY_BOOLEAN', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_TINYINT', ['ARRAY_TINYINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_INT', ['ARRAY_INT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_BIGINT', ['ARRAY_BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DATETIME', ['ARRAY_DATETIME', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DATE', ['ARRAY_DATE', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DATETIMEV2', ['ARRAY_DATETIMEV2', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DATEV2', ['ARRAY_DATEV2', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_FLOAT', ['ARRAY_FLOAT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DOUBLE', ['ARRAY_DOUBLE', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_VARCHAR', ['ARRAY_VARCHAR', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_STRING', ['ARRAY_STRING', 'BIGINT'], ''],

        [['array_slice', '%element_slice%'], 'ARRAY_BOOLEAN', ['ARRAY_BOOLEAN', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_TINYINT', ['ARRAY_TINYINT', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_INT', ['ARRAY_INT', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_BIGINT', ['ARRAY_BIGINT', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DATETIME', ['ARRAY_DATETIME', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DATE', ['ARRAY_DATE', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DATETIMEV2', ['ARRAY_DATETIMEV2', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DATEV2', ['ARRAY_DATEV2', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_FLOAT', ['ARRAY_FLOAT', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DOUBLE', ['ARRAY_DOUBLE', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_VARCHAR', ['ARRAY_VARCHAR', 'BIGINT', 'BIGINT'], ''],
        [['array_slice', '%element_slice%'], 'ARRAY_STRING', ['ARRAY_STRING', 'BIGINT', 'BIGINT'], ''],

        [['array_popback'], 'ARRAY_BOOLEAN', ['ARRAY_BOOLEAN'], ''],
        [['array_popback'], 'ARRAY_TINYINT', ['ARRAY_TINYINT'], ''],
        [['array_popback'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT'], ''],
        [['array_popback'], 'ARRAY_INT', ['ARRAY_INT'], ''],
        [['array_popback'], 'ARRAY_BIGINT', ['ARRAY_BIGINT'], ''],
        [['array_popback'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT'], ''],
        [['array_popback'], 'ARRAY_DATETIME', ['ARRAY_DATETIME'], ''],
        [['array_popback'], 'ARRAY_DATE', ['ARRAY_DATE'], ''],
        [['array_popback'], 'ARRAY_DATETIMEV2', ['ARRAY_DATETIMEV2'], ''],
        [['array_popback'], 'ARRAY_DATEV2', ['ARRAY_DATEV2'], ''],
        [['array_popback'], 'ARRAY_FLOAT', ['ARRAY_FLOAT'], ''],
        [['array_popback'], 'ARRAY_DOUBLE', ['ARRAY_DOUBLE'], ''],
        [['array_popback'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], ''],
        [['array_popback'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], ''],
        [['array_popback'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], ''],
        [['array_popback'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],
        [['array_popback'], 'ARRAY_VARCHAR', ['ARRAY_VARCHAR'], ''],
        [['array_popback'], 'ARRAY_STRING', ['ARRAY_STRING'], ''],

        [['array_popfront'], 'ARRAY_BOOLEAN', ['ARRAY_BOOLEAN'], ''],
        [['array_popfront'], 'ARRAY_TINYINT', ['ARRAY_TINYINT'], ''],
        [['array_popfront'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT'], ''],
        [['array_popfront'], 'ARRAY_INT', ['ARRAY_INT'], ''],
        [['array_popfront'], 'ARRAY_BIGINT', ['ARRAY_BIGINT'], ''],
        [['array_popfront'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT'], ''],
        [['array_popfront'], 'ARRAY_DATETIME', ['ARRAY_DATETIME'], ''],
        [['array_popfront'], 'ARRAY_DATE', ['ARRAY_DATE'], ''],
        [['array_popfront'], 'ARRAY_DATETIMEV2', ['ARRAY_DATETIMEV2'], ''],
        [['array_popfront'], 'ARRAY_DATEV2', ['ARRAY_DATEV2'], ''],
        [['array_popfront'], 'ARRAY_FLOAT', ['ARRAY_FLOAT'], ''],
        [['array_popfront'], 'ARRAY_DOUBLE', ['ARRAY_DOUBLE'], ''],
        [['array_popfront'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], ''],
        [['array_popfront'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], ''],
        [['array_popfront'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], ''],
        [['array_popfront'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],
        [['array_popfront'], 'ARRAY_VARCHAR', ['ARRAY_VARCHAR'], ''],
        [['array_popfront'], 'ARRAY_STRING', ['ARRAY_STRING'], ''],
        [['array_filter'], 'ARRAY_BOOLEAN',['ARRAY_BOOLEAN', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_TINYINT',['ARRAY_TINYINT', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_SMALLINT',['ARRAY_SMALLINT', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_INT',['ARRAY_INT', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_BIGINT',['ARRAY_BIGINT', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_LARGEINT',['ARRAY_LARGEINT', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_FLOAT',['ARRAY_FLOAT', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_DOUBLE',['ARRAY_DOUBLE', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_VARCHAR',['ARRAY_VARCHAR', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_STRING',['ARRAY_STRING', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_DECIMALV2',['ARRAY_DECIMALV2', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_DECIMAL32',['ARRAY_DECIMAL32', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_DECIMAL64',['ARRAY_DECIMAL64', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_DECIMAL128',['ARRAY_DECIMAL128', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_DATETIME',['ARRAY_DATETIME', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_DATE',['ARRAY_DATE', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_DATETIMEV2',['ARRAY_DATETIMEV2', 'ARRAY_BOOLEAN'], ''],
        [['array_filter'], 'ARRAY_DATEV2',['ARRAY_DATEV2', 'ARRAY_BOOLEAN'], ''],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_BOOLEAN'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_TINYINT'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_SMALLINT'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_INT'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_BIGINT'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_LARGEINT'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_FLOAT'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_DOUBLE'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_VARCHAR'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_STRING'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_DECIMALV2'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_DECIMAL32'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_DECIMAL64'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_DECIMAL128'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_DATETIME'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_DATE'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_DATETIMEV2'], 'CUSTOM', ['K']],
        [['array_sortby'], 'ARRAY<K>',['ARRAY<K>', 'ARRAY_DATEV2'], 'CUSTOM', ['K']],

        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_BOOLEAN'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_TINYINT'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_SMALLINT'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_INT'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_BIGINT'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_LARGEINT'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_DATETIME'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_DATE'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_DATETIMEV2'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_DATEV2'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_FLOAT'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_DOUBLE'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_DECIMALV2'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_DECIMAL32'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_DECIMAL64'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_DECIMAL128'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_VARCHAR'], ''],
        [['array_exists'], 'ARRAY_BOOLEAN', ['ARRAY_STRING'], ''],

        [['array_first_index'], 'BIGINT', ['ARRAY_BOOLEAN'], 'ALWAYS_NOT_NULLABLE'],
        [['array_last_index'], 'BIGINT', ['ARRAY_BOOLEAN'], 'ALWAYS_NOT_NULLABLE'],
        [['array_count'], 'BIGINT', ['ARRAY_BOOLEAN'], 'ALWAYS_NOT_NULLABLE'],

        [['array_shuffle', 'shuffle'], 'ARRAY_BOOLEAN',   ['ARRAY_BOOLEAN'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_TINYINT',   ['ARRAY_TINYINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_SMALLINT',  ['ARRAY_SMALLINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_INT',       ['ARRAY_INT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_BIGINT',    ['ARRAY_BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_LARGEINT',  ['ARRAY_LARGEINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DATE',      ['ARRAY_DATE'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_STRING',    ['ARRAY_STRING'], ''],

        [['array_shuffle', 'shuffle'], 'ARRAY_BOOLEAN',   ['ARRAY_BOOLEAN', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_TINYINT',   ['ARRAY_TINYINT', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_SMALLINT',  ['ARRAY_SMALLINT', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_INT',       ['ARRAY_INT', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_BIGINT',    ['ARRAY_BIGINT', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_LARGEINT',  ['ARRAY_LARGEINT', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DATE',      ['ARRAY_DATE', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR', 'BIGINT'], ''],
        [['array_shuffle', 'shuffle'], 'ARRAY_STRING',    ['ARRAY_STRING', 'BIGINT'], ''],

        [['array_pushfront'], 'ARRAY_BOOLEAN',    ['ARRAY_BOOLEAN', 'BOOLEAN'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_TINYINT',    ['ARRAY_TINYINT', 'TINYINT'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_SMALLINT',   ['ARRAY_SMALLINT', 'SMALLINT'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_INT',        ['ARRAY_INT', 'INT'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_BIGINT',     ['ARRAY_BIGINT', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_LARGEINT',   ['ARRAY_LARGEINT', 'LARGEINT'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_DATETIME',   ['ARRAY_DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_DATE',       ['ARRAY_DATE', 'DATE'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_DATETIMEV2', ['ARRAY_DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_DATEV2',     ['ARRAY_DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_FLOAT',      ['ARRAY_FLOAT', 'FLOAT'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_DOUBLE',     ['ARRAY_DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_DECIMALV2',  ['ARRAY_DECIMALV2', 'DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_DECIMAL32',  ['ARRAY_DECIMAL32', 'DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_DECIMAL64',  ['ARRAY_DECIMAL64', 'DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_VARCHAR',    ['ARRAY_VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['array_pushfront'], 'ARRAY_STRING',     ['ARRAY_STRING', 'STRING'], 'ALWAYS_NULLABLE'],

        [['array_pushback'], 'ARRAY<T>',    ['ARRAY<T>', 'T'], 'ALWAYS_NULLABLE', ['T']],
        [['array_pushback'], 'ARRAY_BOOLEAN',    ['ARRAY_BOOLEAN', 'BOOLEAN'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_TINYINT',    ['ARRAY_TINYINT', 'TINYINT'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_SMALLINT',   ['ARRAY_SMALLINT', 'SMALLINT'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_INT',        ['ARRAY_INT', 'INT'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_BIGINT',     ['ARRAY_BIGINT', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_LARGEINT',   ['ARRAY_LARGEINT', 'LARGEINT'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_DATETIME',   ['ARRAY_DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_DATE',       ['ARRAY_DATE', 'DATE'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_DATETIMEV2', ['ARRAY_DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_DATEV2',     ['ARRAY_DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_FLOAT',      ['ARRAY_FLOAT', 'FLOAT'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_DOUBLE',     ['ARRAY_DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_DECIMALV2',  ['ARRAY_DECIMALV2', 'DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_DECIMAL32',  ['ARRAY_DECIMAL32', 'DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_DECIMAL64',  ['ARRAY_DECIMAL64', 'DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128', 'DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_VARCHAR',    ['ARRAY_VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['array_pushback'], 'ARRAY_STRING',     ['ARRAY_STRING', 'STRING'], 'ALWAYS_NULLABLE'],

        [['array_with_constant'], 'ARRAY_BOOLEAN', ['BIGINT', 'BOOLEAN'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_TINYINT', ['BIGINT', 'TINYINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_SMALLINT', ['BIGINT','SMALLINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_INT', ['BIGINT', 'INT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_BIGINT', ['BIGINT', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_LARGEINT', ['BIGINT', 'LARGEINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_DATETIME', ['BIGINT', 'DATETIME'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_DATE', ['BIGINT', 'DATE'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_DATETIMEV2', ['BIGINT', 'DATETIMEV2'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_DATEV2', ['BIGINT', 'DATEV2'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_FLOAT', ['BIGINT', 'FLOAT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_DOUBLE', ['BIGINT', 'DOUBLE'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_DECIMALV2', ['BIGINT', 'DECIMALV2'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_DECIMAL32', ['BIGINT', 'DECIMAL32'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_DECIMAL64', ['BIGINT', 'DECIMAL64'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_DECIMAL128', ['BIGINT', 'DECIMAL128'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_VARCHAR', ['BIGINT', 'VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['array_with_constant'], 'ARRAY_STRING', ['BIGINT', 'STRING'], 'ALWAYS_NOT_NULLABLE'],

        [['array_repeat'], 'ARRAY_BOOLEAN', ['BOOLEAN', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_TINYINT', ['TINYINT', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_SMALLINT', ['SMALLINT', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_INT', ['INT', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_BIGINT', ['BIGINT', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_LARGEINT', ['LARGEINT', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_DATETIME', ['DATETIME', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_DATE', ['DATE', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_DATETIMEV2', ['DATETIMEV2', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_DATEV2', ['DATEV2', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_FLOAT', ['FLOAT', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_DOUBLE', ['DOUBLE', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_DECIMALV2', ['DECIMALV2', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_DECIMAL32', ['DECIMAL32', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_DECIMAL64', ['DECIMAL64', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_DECIMAL128', ['DECIMAL128', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_VARCHAR', ['VARCHAR', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['array_repeat'], 'ARRAY_STRING', ['STRING', 'BIGINT'], 'ALWAYS_NOT_NULLABLE'],

        [['array_range', 'sequence'], 'ARRAY_INT', ['INT'], 'ALWAYS_NULLABLE'],
        [['array_range', 'sequence'], 'ARRAY_INT', ['INT', 'INT'], 'ALWAYS_NULLABLE'],
        [['array_range', 'sequence'], 'ARRAY_INT', ['INT', 'INT', 'INT'], 'ALWAYS_NULLABLE'],
        [['array_range', 'sequence'], 'ARRAY_DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['array_range', 'sequence'], 'ARRAY_DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],

        [['array_zip'], 'ARRAY', ['ARRAY<T>', '...'], '', ['T']],

        [['array_split'], 'ARRAY_BOOLEAN',['ARRAY_BOOLEAN', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_TINYINT',['ARRAY_TINYINT', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_SMALLINT',['ARRAY_SMALLINT', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_INT',['ARRAY_INT', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_BIGINT',['ARRAY_BIGINT', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_LARGEINT',['ARRAY_LARGEINT', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_FLOAT',['ARRAY_FLOAT', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_DOUBLE',['ARRAY_DOUBLE', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_VARCHAR',['ARRAY_VARCHAR', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_STRING',['ARRAY_STRING', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_DECIMALV2',['ARRAY_DECIMALV2', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_DECIMAL32',['ARRAY_DECIMAL32', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_DECIMAL64',['ARRAY_DECIMAL64', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_DECIMAL128',['ARRAY_DECIMAL128', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_DATETIME',['ARRAY_DATETIME', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_DATE',['ARRAY_DATE', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_DATETIMEV2',['ARRAY_DATETIMEV2', 'ARRAY_BOOLEAN'], ''],
        [['array_split'], 'ARRAY_DATEV2',['ARRAY_DATEV2', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_BOOLEAN',['ARRAY_BOOLEAN', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_TINYINT',['ARRAY_TINYINT', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_SMALLINT',['ARRAY_SMALLINT', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_INT',['ARRAY_INT', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_BIGINT',['ARRAY_BIGINT', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_LARGEINT',['ARRAY_LARGEINT', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_FLOAT',['ARRAY_FLOAT', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_DOUBLE',['ARRAY_DOUBLE', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_VARCHAR',['ARRAY_VARCHAR', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_STRING',['ARRAY_STRING', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_DECIMALV2',['ARRAY_DECIMALV2', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_DECIMAL32',['ARRAY_DECIMAL32', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_DECIMAL64',['ARRAY_DECIMAL64', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_DECIMAL128',['ARRAY_DECIMAL128', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_DATETIME',['ARRAY_DATETIME', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_DATE',['ARRAY_DATE', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_DATETIMEV2',['ARRAY_DATETIMEV2', 'ARRAY_BOOLEAN'], ''],
        [['array_reverse_split'], 'ARRAY_DATEV2',['ARRAY_DATEV2', 'ARRAY_BOOLEAN'], ''],


        # reverse function for string builtin
        [['reverse'], 'VARCHAR', ['VARCHAR'], ''],
        # reverse function support the longtext
        [['reverse'], 'STRING', ['STRING'], ''],
        # reverse function support the array type
        [['reverse'], 'ARRAY_TINYINT',   ['ARRAY_TINYINT'], ''],
        [['reverse'], 'ARRAY_SMALLINT',  ['ARRAY_SMALLINT'], ''],
        [['reverse'], 'ARRAY_INT',       ['ARRAY_INT'], ''],
        [['reverse'], 'ARRAY_BIGINT',    ['ARRAY_BIGINT'], ''],
        [['reverse'], 'ARRAY_LARGEINT',  ['ARRAY_LARGEINT'], ''],
        [['reverse'], 'ARRAY_DATETIME',  ['ARRAY_DATETIME'], ''],
        [['reverse'], 'ARRAY_DATE',      ['ARRAY_DATE'], ''],
        [['reverse'], 'ARRAY_DATETIMEV2',  ['ARRAY_DATETIMEV2'], ''],
        [['reverse'], 'ARRAY_DATEV2',      ['ARRAY_DATEV2'], ''],
        [['reverse'], 'ARRAY_FLOAT',     ['ARRAY_FLOAT'], ''],
        [['reverse'], 'ARRAY_DOUBLE',    ['ARRAY_DOUBLE'], ''],
        [['reverse'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], ''],
        [['reverse'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], ''],
        [['reverse'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], ''],
        [['reverse'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], ''],
        [['reverse'], 'ARRAY_VARCHAR',   ['ARRAY_VARCHAR'], ''],
        [['reverse'], 'ARRAY_STRING',    ['ARRAY_STRING'], '']
    ],

    # Timestamp functions
    "Timestamp": [
        [['unix_timestamp'], 'INT', [], 'ALWAYS_NOT_NULLABLE'],
        [['unix_timestamp'], 'INT', ['DATETIME'], 'DEPEND_ON_ARGUMENT'],
        [['unix_timestamp'], 'INT', ['DATE'], 'DEPEND_ON_ARGUMENT'],
        [['unix_timestamp'], 'DECIMAL64', ['DATETIMEV2'], 'DEPEND_ON_ARGUMENT'],
        [['unix_timestamp'], 'INT', ['DATEV2'], 'DEPEND_ON_ARGUMENT'],
        [['unix_timestamp'], 'DECIMAL64', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['unix_timestamp'], 'DECIMAL64', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['from_unixtime'], 'VARCHAR', ['BIGINT'], 'ALWAYS_NULLABLE'],
        [['from_unixtime'], 'VARCHAR', ['BIGINT', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['from_unixtime'], 'VARCHAR', ['BIGINT', 'STRING'], 'ALWAYS_NULLABLE'],
        [['from_microsecond'], 'DATETIMEV2', ['BIGINT'], 'ALWAYS_NULLABLE'],
        [['from_millisecond'], 'DATETIMEV2', ['BIGINT'], 'ALWAYS_NULLABLE'],
        [['from_second'], 'DATETIMEV2', ['BIGINT'], 'ALWAYS_NULLABLE'],
        [['second_timestamp'], 'BIGINT', ['DATETIMEV2'], 'DEPEND_ON_ARGUMENT'],
        [['millisecond_timestamp'], 'BIGINT', ['DATETIMEV2'], 'DEPEND_ON_ARGUMENT'],
        [['microsecond_timestamp'], 'BIGINT', ['DATETIMEV2'], 'DEPEND_ON_ARGUMENT'],
        [['now', 'current_timestamp', 'localtime', 'localtimestamp'], 'DATETIME', [], 'DEPEND_ON_ARGUMENT'],
        [['now', 'current_timestamp', 'localtime', 'localtimestamp'], 'DATETIMEV2', ['INT'], 'DEPEND_ON_ARGUMENT'],
        [['curtime', 'current_time'], 'TIMEV2', [], 'ALWAYS_NOT_NULLABLE'],
        [['curdate', 'current_date'], 'DATE', [], 'ALWAYS_NOT_NULLABLE'],
        [['utc_timestamp'], 'DATETIME', [], 'ALWAYS_NOT_NULLABLE'],
        [['timestamp'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],

        [['from_days'], 'DATEV2', ['INT'], 'ALWAYS_NULLABLE'],
        [['from_days'], 'DATE', ['INT'], 'ALWAYS_NULLABLE'],
        [['last_day'], 'DATE', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['last_day'], 'DATE', ['DATE'], 'ALWAYS_NULLABLE'],
        [['last_day'], 'DATEV2', ['DATETIMEV2'], ''],
        [['last_day'], 'DATEV2', ['DATEV2'], ''],
        [['to_monday'], 'DATEV2', ['DATETIMEV2'], ''],
        [['to_monday'], 'DATEV2', ['DATEV2'], ''],
        [['to_monday'], 'DATE', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['to_monday'], 'DATE', ['DATE'], 'ALWAYS_NULLABLE'],
        [['to_days'], 'INT', ['DATE'], 'ALWAYS_NULLABLE'],


        [['date_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['date_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['date_trunc'], 'DATETIMEV2', ['DATETIMEV2', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['date_trunc'], 'DATETIME', ['DATETIME', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['date_trunc'], 'DATEV2', ['DATEV2', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['date_trunc'], 'DATE', ['DATE', 'VARCHAR'], 'ALWAYS_NULLABLE'],

        [['year'], 'SMALLINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['month'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['quarter'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['dayofweek'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['weekday'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['day', 'dayofmonth'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['dayofyear'], 'SMALLINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['weekofyear'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['yearweek'], 'INT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['yearweek'], 'INT', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['week'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['week'], 'TINYINT', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['hour'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['minute'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['second'], 'TINYINT', ['DATETIME'], 'ALWAYS_NULLABLE'],

        [['makedate'], 'DATE', ['INT', 'INT'], 'ALWAYS_NULLABLE'],
        [['years_add'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['years_sub'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['months_add', 'add_months'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['months_sub'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['weeks_add'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['weeks_sub'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['days_add', 'date_add', 'adddate'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['days_sub', 'date_sub', 'subdate'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['hours_add'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['hours_sub'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['minutes_add'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['minutes_sub'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['seconds_add'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['seconds_sub'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],

        [['datediff'], 'INT', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['timediff'], 'TIMEV2', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],

        [['date_format'], 'VARCHAR', ['DATETIME', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['date_format'], 'VARCHAR', ['DATE', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['date', 'to_date'], 'DATE', ['DATETIME'], 'ALWAYS_NULLABLE'],

        [['dayname'], 'VARCHAR', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['monthname'], 'VARCHAR', ['DATETIME'], 'ALWAYS_NULLABLE'],

        [['convert_tz'], 'DATETIME', ['DATETIME', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],

        [['years_diff'], 'BIGINT', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['months_diff'], 'BIGINT', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['weeks_diff'], 'BIGINT', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['days_diff'], 'BIGINT', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['hours_diff'], 'BIGINT', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['minutes_diff'], 'BIGINT', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['seconds_diff'], 'BIGINT', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],

        [['year_floor'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['year_floor'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['year_floor'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['year_floor'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['hour_floor'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['hour_floor'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['hour_floor'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['hour_floor'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['hour_ceil'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['hour_ceil'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['hour_ceil'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['hour_ceil'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['minute_floor'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['minute_floor'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['minute_floor'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['minute_floor'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['minute_ceil'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['minute_ceil'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['minute_ceil'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['minute_ceil'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['second_floor'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['second_floor'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['second_floor'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['second_floor'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['second_ceil'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['second_ceil'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['second_ceil'], 'DATETIME', ['DATETIME', 'INT'], 'ALWAYS_NULLABLE'],
        [['second_ceil'], 'DATETIME', ['DATETIME', 'INT', 'DATETIME'], 'ALWAYS_NULLABLE'],

        [['timestamp'], 'DATETIMEV2', ['DATETIMEV2'], ''],

        [['to_days'], 'INT', ['DATEV2'], ''],
        [['time_to_sec'], 'INT', ['TIMEV2'], ''],
        [['time_to_sec'], 'INT', ['TIMEV2'], ''],
        [['sec_to_time'], 'TIMEV2', ['INT'], ''],

        [['year'], 'SMALLINT', ['DATETIMEV2'], ''],
        [['month'], 'TINYINT', ['DATETIMEV2'], ''],
        [['quarter'], 'TINYINT', ['DATETIMEV2'], ''],
        [['dayofweek'], 'TINYINT', ['DATETIMEV2'], ''],
        [['weekday'], 'TINYINT', ['DATETIMEV2'], ''],
        [['day', 'dayofmonth'], 'TINYINT', ['DATETIMEV2'], ''],
        [['dayofyear'], 'SMALLINT', ['DATETIMEV2'], ''],
        [['weekofyear'], 'TINYINT', ['DATETIMEV2'], ''],
        [['yearweek'], 'INT', ['DATETIMEV2'], ''],
        [['yearweek'], 'INT', ['DATETIMEV2', 'INT'], ''],
        [['week'], 'TINYINT', ['DATETIMEV2'], ''],
        [['week'], 'TINYINT', ['DATETIMEV2', 'INT'], ''],
        [['hour'], 'TINYINT', ['DATETIMEV2'], ''],
        [['minute'], 'TINYINT', ['DATETIMEV2'], ''],
        [['second'], 'TINYINT', ['DATETIMEV2'], ''],
        [['microsecond'], 'INT', ['DATETIMEV2'], ''],

        [['year'], 'SMALLINT', ['DATEV2'], ''],
        [['month'], 'TINYINT', ['DATEV2'], ''],
        [['quarter'], 'TINYINT', ['DATEV2'], ''],
        [['dayofweek'], 'TINYINT', ['DATEV2'], ''],
        [['weekday'], 'TINYINT', ['DATEV2'], ''],
        [['day', 'dayofmonth'], 'TINYINT', ['DATEV2'], ''],
        [['dayofyear'], 'SMALLINT', ['DATEV2'], ''],
        [['weekofyear'], 'TINYINT', ['DATEV2'], ''],
        [['yearweek'], 'INT', ['DATEV2'], ''],
        [['yearweek'], 'INT', ['DATEV2', 'INT'], ''],
        [['week'], 'TINYINT', ['DATEV2'], ''],
        [['week'], 'TINYINT', ['DATEV2', 'INT'], ''],
        [['hour'], 'TINYINT', ['DATEV2'], ''],
        [['minute'], 'TINYINT', ['DATEV2'], ''],
        [['second'], 'TINYINT', ['DATEV2'], ''],

        [['years_add'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['years_sub'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['months_add', 'add_months'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['months_sub'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['weeks_add'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['weeks_sub'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['days_add', 'date_add', 'adddate'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['days_sub', 'date_sub', 'subdate'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['hours_add'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['hours_sub'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['minutes_add'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['minutes_sub'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['seconds_add'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['seconds_sub'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['microseconds_add'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['microseconds_sub'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['milliseconds_add'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],
        [['milliseconds_sub'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], ''],

        [['years_add'], 'DATEV2', ['DATEV2', 'INT'], ''],
        [['years_sub'], 'DATEV2', ['DATEV2', 'INT'], ''],
        [['months_add', 'add_months'], 'DATEV2', ['DATEV2', 'INT'], ''],
        [['months_sub'], 'DATEV2', ['DATEV2', 'INT'], ''],
        [['weeks_add'], 'DATEV2', ['DATEV2', 'INT'], ''],
        [['weeks_sub'], 'DATEV2', ['DATEV2', 'INT'], ''],
        [['days_add', 'date_add', 'adddate'], 'DATEV2', ['DATEV2', 'INT'], ''],
        [['days_sub', 'date_sub', 'subdate'], 'DATEV2', ['DATEV2', 'INT'], ''],
        [['days_add', 'date_add', 'adddate'], 'DATE', ['DATE', 'INT'], ''],
        [['days_sub', 'date_sub', 'subdate'], 'DATE', ['DATE', 'INT'], ''],
        [['hours_add'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],
        [['hours_sub'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],
        [['minutes_add'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],
        [['minutes_sub'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],
        [['seconds_add'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],
        [['seconds_sub'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],
        [['microseconds_add'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],
        [['microseconds_sub'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],
        [['milliseconds_add'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],
        [['milliseconds_sub'], 'DATETIMEV2', ['DATEV2', 'INT'], ''],

        [['datediff'], 'INT', ['DATETIMEV2', 'DATETIMEV2'], ''],
        [['timediff'], 'TIMEV2', ['DATETIMEV2', 'DATETIMEV2'], ''],

        [['datediff'], 'INT', ['DATETIMEV2', 'DATEV2'], ''],
        [['timediff'], 'TIMEV2', ['DATETIMEV2', 'DATEV2'], ''],

        [['datediff'], 'INT', ['DATEV2', 'DATETIMEV2'], ''],
        [['timediff'], 'TIMEV2', ['DATEV2', 'DATETIMEV2'], ''],

        [['datediff'], 'INT', ['DATEV2', 'DATEV2'], ''],
        [['timediff'], 'TIMEV2', ['DATEV2', 'DATEV2'], ''],

        [['str_to_date'], 'DATETIMEV2', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['str_to_date'], 'DATETIMEV2', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['date_format'], 'VARCHAR', ['DATETIMEV2', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['date_format'], 'VARCHAR', ['DATEV2', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['date', 'to_date', 'datev2', 'to_datev2'], 'DATEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],

        [['dayname'], 'VARCHAR', ['DATETIMEV2'], ''],
        [['monthname'], 'VARCHAR', ['DATETIMEV2'], ''],

        [['dayname'], 'VARCHAR', ['DATEV2'], ''],
        [['monthname'], 'VARCHAR', ['DATEV2'], ''],

        [['convert_tz'], 'DATETIMEV2', ['DATETIMEV2', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],

        [['years_diff'], 'BIGINT', ['DATETIMEV2', 'DATETIMEV2'], ''],
        [['months_diff'], 'BIGINT', ['DATETIMEV2', 'DATETIMEV2'], ''],
        [['weeks_diff'], 'BIGINT', ['DATETIMEV2', 'DATETIMEV2'], ''],
        [['days_diff'], 'BIGINT', ['DATETIMEV2', 'DATETIMEV2'], ''],
        [['hours_diff'], 'BIGINT', ['DATETIMEV2', 'DATETIMEV2'], ''],
        [['minutes_diff'], 'BIGINT', ['DATETIMEV2', 'DATETIMEV2'], ''],
        [['seconds_diff'], 'BIGINT', ['DATETIMEV2', 'DATETIMEV2'], ''],
        [['microseconds_diff'], 'BIGINT', ['DATETIMEV2', 'DATETIMEV2'], ''],
        [['milliseconds_diff'], 'BIGINT', ['DATETIMEV2', 'DATETIMEV2'], ''],

        [['years_diff'], 'BIGINT', ['DATEV2', 'DATETIMEV2'], ''],
        [['months_diff'], 'BIGINT', ['DATEV2', 'DATETIMEV2'], ''],
        [['weeks_diff'], 'BIGINT', ['DATEV2', 'DATETIMEV2'], ''],
        [['days_diff'], 'BIGINT', ['DATEV2', 'DATETIMEV2'], ''],
        [['hours_diff'], 'BIGINT', ['DATEV2', 'DATETIMEV2'], ''],
        [['minutes_diff'], 'BIGINT', ['DATEV2', 'DATETIMEV2'], ''],
        [['seconds_diff'], 'BIGINT', ['DATEV2', 'DATETIMEV2'], ''],
        [['microseconds_diff'], 'BIGINT', ['DATEV2', 'DATETIMEV2'], ''],
        [['milliseconds_diff'], 'BIGINT', ['DATEV2', 'DATETIMEV2'], ''],

        [['years_diff'], 'BIGINT', ['DATETIMEV2', 'DATEV2'], ''],
        [['months_diff'], 'BIGINT', ['DATETIMEV2', 'DATEV2'], ''],
        [['weeks_diff'], 'BIGINT', ['DATETIMEV2', 'DATEV2'], ''],
        [['days_diff'], 'BIGINT', ['DATETIMEV2', 'DATEV2'], ''],
        [['hours_diff'], 'BIGINT', ['DATETIMEV2', 'DATEV2'], ''],
        [['minutes_diff'], 'BIGINT', ['DATETIMEV2', 'DATEV2'], ''],
        [['seconds_diff'], 'BIGINT', ['DATETIMEV2', 'DATEV2'], ''],
        [['microseconds_diff'], 'BIGINT', ['DATETIMEV2', 'DATEV2'], ''],
        [['milliseconds_diff'], 'BIGINT', ['DATETIMEV2', 'DATEV2'], ''],

        [['years_diff'], 'BIGINT', ['DATEV2', 'DATEV2'], ''],
        [['months_diff'], 'BIGINT', ['DATEV2', 'DATEV2'], ''],
        [['weeks_diff'], 'BIGINT', ['DATEV2', 'DATEV2'], ''],
        [['days_diff'], 'BIGINT', ['DATEV2', 'DATEV2'], ''],
        [['hours_diff'], 'BIGINT', ['DATEV2', 'DATEV2'], ''],
        [['minutes_diff'], 'BIGINT', ['DATEV2', 'DATEV2'], ''],
        [['seconds_diff'], 'BIGINT', ['DATEV2', 'DATEV2'], ''],
        [['microseconds_diff'], 'BIGINT', ['DATEV2', 'DATEV2'], ''],
        [['milliseconds_diff'], 'BIGINT', ['DATEV2', 'DATEV2'], ''],

        [['year_floor'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['year_floor'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['year_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['year_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['hour_floor'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['hour_floor'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['hour_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['hour_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['hour_ceil'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['hour_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['hour_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['hour_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['minute_floor'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['minute_floor'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['minute_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['minute_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['minute_ceil'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['minute_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['minute_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['minute_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['second_floor'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['second_floor'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['second_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['second_floor'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['second_ceil'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['second_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['second_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['second_ceil'], 'DATETIMEV2', ['DATETIMEV2', 'INT', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],

        [['year_floor'], 'DATEV2', ['DATEV2'], 'ALWAYS_NULLABLE'],
        [['year_floor'], 'DATEV2', ['DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['year_floor'], 'DATEV2', ['DATEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['year_floor'], 'DATEV2', ['DATEV2', 'INT', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATEV2', ['DATEV2'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATEV2', ['DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATEV2', ['DATEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['year_ceil'], 'DATEV2', ['DATEV2', 'INT', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATEV2', ['DATEV2'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATEV2', ['DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATEV2', ['DATEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['month_floor'], 'DATEV2', ['DATEV2', 'INT', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATEV2', ['DATEV2'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATEV2', ['DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATEV2', ['DATEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['month_ceil'], 'DATEV2', ['DATEV2', 'INT', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATEV2', ['DATEV2'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATEV2', ['DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATEV2', ['DATEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['week_floor'], 'DATEV2', ['DATEV2', 'INT', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATEV2', ['DATEV2'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATEV2', ['DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATEV2', ['DATEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['week_ceil'], 'DATEV2', ['DATEV2', 'INT', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATEV2', ['DATEV2'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATEV2', ['DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATEV2', ['DATEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['day_floor'], 'DATEV2', ['DATEV2', 'INT', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATEV2', ['DATEV2'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATEV2', ['DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATEV2', ['DATEV2', 'INT'], 'ALWAYS_NULLABLE'],
        [['day_ceil'], 'DATEV2', ['DATEV2', 'INT', 'DATEV2'], 'ALWAYS_NULLABLE']
    ],


    # Math builtin functions
    # Please make sure functions are ordered by alphabetical order
    "Math": [
        [['abs'], 'DOUBLE', ['DOUBLE'], ''],
        [['abs'], 'FLOAT', ['FLOAT'], ''],
        [['abs'], 'LARGEINT', ['LARGEINT'], ''],
        [['abs'], 'LARGEINT', ['BIGINT'], ''],
        [['abs'], 'INT', ['SMALLINT'], ''],
        [['abs'], 'BIGINT', ['INT'], ''],
        [['abs'], 'SMALLINT', ['TINYINT'], ''],
        [['abs'], 'DECIMALV2', ['DECIMALV2'], ''],
        [['abs'], 'DECIMAL32', ['DECIMAL32'], ''],
        [['abs'], 'DECIMAL64', ['DECIMAL64'], ''],
        [['abs'], 'DECIMAL128', ['DECIMAL128'], ''],
        [['acos'], 'DOUBLE', ['DOUBLE'], ''],
        [['atan'], 'DOUBLE', ['DOUBLE'], ''],
        [['atan2'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], ''],
        [['asin'], 'DOUBLE', ['DOUBLE'], ''],

        [['bin'], 'VARCHAR', ['BIGINT'], ''],
              
        [['cbrt'], 'DOUBLE', ['DOUBLE'], ''],
        [['ceil', 'ceiling', 'dceil'], 'DOUBLE', ['DOUBLE'], ''],
        [['ceil', 'ceiling', 'dceil'], 'DOUBLE', ['DOUBLE', 'INT'], ''],
        [['ceil', 'ceiling', 'dceil'], 'DECIMAL32', ['DECIMAL32'], ''],
        [['ceil', 'ceiling', 'dceil'], 'DECIMAL64', ['DECIMAL64'], ''],
        [['ceil', 'ceiling', 'dceil'], 'DECIMAL128', ['DECIMAL128'], ''],
        [['ceil', 'dceil'], 'DECIMAL32', ['DECIMAL32', 'INT'], ''],
        [['ceil', 'dceil'], 'DECIMAL64', ['DECIMAL64', 'INT'], ''],
        [['ceil', 'dceil'], 'DECIMAL128', ['DECIMAL128', 'INT'], ''],
        [['conv'], 'VARCHAR', ['BIGINT', 'TINYINT', 'TINYINT'], 'ALWAYS_NULLABLE'],
        [['conv'], 'VARCHAR', ['VARCHAR', 'TINYINT', 'TINYINT'], 'ALWAYS_NULLABLE'],
        [['conv'], 'VARCHAR', ['STRING', 'TINYINT', 'TINYINT'], 'ALWAYS_NULLABLE'],
        [['cos'], 'DOUBLE', ['DOUBLE'], ''],
        [['cosh'], 'DOUBLE', ['DOUBLE'], ''],

        [['degrees'], 'DOUBLE', ['DOUBLE'], ''],

        [['e'], 'DOUBLE', [], 'ALWAYS_NOT_NULLABLE'],
        [['exp', 'dexp'], 'DOUBLE', ['DOUBLE'], ''],

        [['floor', 'dfloor'], 'DOUBLE', ['DOUBLE'], ''],
        [['floor', 'dfloor'], 'DOUBLE', ['DOUBLE', 'INT'], ''],
        [['floor', 'dfloor'], 'DECIMAL32', ['DECIMAL32'], ''],
        [['floor', 'dfloor'], 'DECIMAL32', ['DECIMAL32', 'INT'], ''],
        [['floor', 'dfloor'], 'DECIMAL64', ['DECIMAL64'], ''],
        [['floor', 'dfloor'], 'DECIMAL64', ['DECIMAL64', 'INT'], ''],
        [['floor', 'dfloor'], 'DECIMAL128', ['DECIMAL128'], ''],
        [['floor', 'dfloor'], 'DECIMAL128', ['DECIMAL128', 'INT'], ''],

        [['greatest'], 'TINYINT', ['TINYINT', '...'], ''],
        [['greatest'], 'SMALLINT', ['SMALLINT', '...'], ''],
        [['greatest'], 'INT', ['INT', '...'], ''],
        [['greatest'], 'BIGINT', ['BIGINT', '...'], ''],
        [['greatest'], 'LARGEINT', ['LARGEINT', '...'], ''],
        [['greatest'], 'FLOAT', ['FLOAT', '...'], ''],
        [['greatest'], 'DOUBLE', ['DOUBLE', '...'], ''],
        [['greatest'], 'DECIMALV2', ['DECIMALV2', '...'], ''],
        [['greatest'], 'DECIMAL32', ['DECIMAL32', '...'], ''],
        [['greatest'], 'DECIMAL64', ['DECIMAL64', '...'], ''],
        [['greatest'], 'DECIMAL128', ['DECIMAL128', '...'], ''],
        [['greatest'], 'DATE', ['DATE', '...'], ''],
        [['greatest'], 'DATEV2', ['DATEV2', '...'], ''],
        [['greatest'], 'DATETIME', ['DATETIME', '...'], ''],
        [['greatest'], 'DATETIMEV2', ['DATETIMEV2', '...'], ''],
        [['greatest'], 'VARCHAR', ['VARCHAR', '...'], ''],
        [['greatest'], 'STRING', ['STRING', '...'], ''],

        [['hex'], 'VARCHAR', ['BIGINT'], ''],
        [['hex'], 'VARCHAR', ['VARCHAR'], ''],
        [['hex'], 'STRING', ['STRING'], ''],

        [['ln', 'dlog1'], 'DOUBLE', ['DOUBLE'], 'ALWAYS_NULLABLE'],
        [['log'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],
        [['log2'], 'DOUBLE', ['DOUBLE'], 'ALWAYS_NULLABLE'],
        [['log10', 'dlog10'], 'DOUBLE', ['DOUBLE'], 'ALWAYS_NULLABLE'],
        [['least'], 'TINYINT', ['TINYINT', '...'], ''],
        [['least'], 'SMALLINT', ['SMALLINT', '...'], ''],
        [['least'], 'INT', ['INT', '...'], ''],
        [['least'], 'BIGINT', ['BIGINT', '...'], ''],
        [['least'], 'LARGEINT', ['LARGEINT', '...'], ''],
        [['least'], 'FLOAT', ['FLOAT', '...'], ''],
        [['least'], 'DOUBLE', ['DOUBLE', '...'], ''],
        [['least'], 'DATE', ['DATE', '...'], ''],
        [['least'], 'DATEV2', ['DATEV2', '...'], ''],
        [['least'], 'DATETIME', ['DATETIME', '...'], ''],
        [['least'], 'DATETIMEV2', ['DATETIMEV2', '...'], ''],
        [['least'], 'DECIMALV2', ['DECIMALV2', '...'], ''],
        [['least'], 'DECIMAL32', ['DECIMAL32', '...'], ''],
        [['least'], 'DECIMAL64', ['DECIMAL64', '...'], ''],
        [['least'], 'DECIMAL128', ['DECIMAL128', '...'], ''],
        [['least'], 'VARCHAR', ['VARCHAR', '...'], ''],
        [['least'], 'STRING', ['STRING', '...'], ''],

        [['mod'], 'TINYINT', ['TINYINT', 'TINYINT'], 'ALWAYS_NULLABLE'],
        [['mod'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], 'ALWAYS_NULLABLE'],
        [['mod'], 'INT', ['INT', 'INT'], 'ALWAYS_NULLABLE'],
        [['mod'], 'BIGINT', ['BIGINT', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['mod'], 'LARGEINT', ['LARGEINT', 'LARGEINT'], 'ALWAYS_NULLABLE'],
        [['mod'], 'DECIMALV2', ['DECIMALV2', 'DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['mod'], 'DECIMAL32', ['DECIMAL32', 'DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['mod'], 'DECIMAL64', ['DECIMAL64', 'DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['mod'], 'DECIMAL128', ['DECIMAL128', 'DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['mod', 'fmod'], 'FLOAT', ['FLOAT', 'FLOAT'], 'ALWAYS_NULLABLE'],
        [['mod', 'fmod'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],

        [['negative'], 'BIGINT', ['BIGINT'], ''],
        [['negative'], 'DOUBLE', ['DOUBLE'], ''],
        [['negative'], 'DECIMALV2', ['DECIMALV2'], ''],
        [['negative'], 'DECIMAL32', ['DECIMAL32'], ''],
        [['negative'], 'DECIMAL64', ['DECIMAL64'], ''],
        [['negative'], 'DECIMAL128', ['DECIMAL128'], ''],

        [['pi'], 'DOUBLE', [], 'ALWAYS_NOT_NULLABLE'],
        [['pmod'], 'BIGINT', ['BIGINT', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['pmod'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],
        [['positive'], 'BIGINT', ['BIGINT'], ''],
        [['positive'], 'DOUBLE', ['DOUBLE'], ''],
        [['positive'], 'DECIMALV2', ['DECIMALV2'], ''],
        [['positive'], 'DECIMAL32', ['DECIMAL32'], ''],
        [['positive'], 'DECIMAL64', ['DECIMAL64'], ''],
        [['positive'], 'DECIMAL128', ['DECIMAL128'], ''],
        [['pow', 'power', 'dpow', 'fpow'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], ''],

        [['radians'], 'DOUBLE', ['DOUBLE'], ''],
        [['rand', 'random'], 'DOUBLE', [], 'ALWAYS_NOT_NULLABLE'],
        [['rand', 'random'], 'DOUBLE', ['BIGINT'], ''],
        [['rand', 'random'], 'BIGINT', ['BIGINT', 'BIGINT'], ''],
        [['round', 'dround'], 'DOUBLE', ['DOUBLE'], ''],
        [['round', 'dround'], 'DOUBLE', ['DOUBLE', 'INT'], ''],
        [['round', 'dround'], 'DECIMAL32', ['DECIMAL32'], ''],
        [['round', 'dround'], 'DECIMAL32', ['DECIMAL32', 'INT'], ''],
        [['round', 'dround'], 'DECIMAL64', ['DECIMAL64'], ''],
        [['round', 'dround'], 'DECIMAL64', ['DECIMAL64', 'INT'], ''],
        [['round', 'dround'], 'DECIMAL128', ['DECIMAL128'], ''],
        [['round', 'dround'], 'DECIMAL128', ['DECIMAL128', 'INT'], ''],
        [['round_bankers'], 'DOUBLE', ['DOUBLE'], ''],
        [['round_bankers'], 'DOUBLE', ['DOUBLE', 'INT'], ''],
        [['round_bankers'], 'DECIMAL32', ['DECIMAL32'], ''],
        [['round_bankers'], 'DECIMAL32', ['DECIMAL32', 'INT'], ''],
        [['round_bankers'], 'DECIMAL64', ['DECIMAL64'], ''],
        [['round_bankers'], 'DECIMAL64', ['DECIMAL64', 'INT'], ''],
        [['round_bankers'], 'DECIMAL128', ['DECIMAL128'], ''],
        [['round_bankers'], 'DECIMAL128', ['DECIMAL128', 'INT'], ''],
        [['round_bankers', 'round_bankers'], 'DOUBLE', ['DOUBLE', 'INT'], ''],

        [['sign'], 'TINYINT', ['DOUBLE'], ''],
        [['sin'], 'DOUBLE', ['DOUBLE'], ''],  
        [['sqrt', 'dsqrt'], 'DOUBLE', ['DOUBLE'], ''],
        
        [['tan'], 'DOUBLE', ['DOUBLE'], ''],
        [['tanh'], 'DOUBLE', ['DOUBLE'], ''],
        [['truncate'], 'DOUBLE', ['DOUBLE'], ''],
        [['truncate'], 'DOUBLE', ['DOUBLE', 'INT'], ''],
        [['truncate'], 'DECIMAL32', ['DECIMAL32', 'INT'], ''],
        [['truncate'], 'DECIMAL64', ['DECIMAL64', 'INT'], ''],
        [['truncate'], 'DECIMAL128', ['DECIMAL128', 'INT'], ''],
        
        [['unhex'], 'VARCHAR', ['VARCHAR'], 'DEPEND_ON_ARGUMENT'],
        [['unhex'], 'STRING', ['STRING'], 'DEPEND_ON_ARGUMENT']
    ],

    # Conditional Functions
    # Some of these have empty symbols because the BE special-cases them based on the
    # function name
    "Conditional": [
        [['if'], 'BOOLEAN', ['BOOLEAN', 'BOOLEAN', 'BOOLEAN'], 'CUSTOM'],
        [['if'], 'TINYINT', ['BOOLEAN', 'TINYINT', 'TINYINT'], 'CUSTOM'],
        [['if'], 'SMALLINT', ['BOOLEAN', 'SMALLINT', 'SMALLINT'], 'CUSTOM'],
        [['if'], 'INT', ['BOOLEAN', 'INT', 'INT'], 'CUSTOM'],
        [['if'], 'BIGINT', ['BOOLEAN', 'BIGINT', 'BIGINT'], 'CUSTOM'],
        [['if'], 'LARGEINT', ['BOOLEAN', 'LARGEINT', 'LARGEINT'], 'CUSTOM'],
        [['if'], 'FLOAT', ['BOOLEAN', 'FLOAT', 'FLOAT'], 'CUSTOM'],
        [['if'], 'DOUBLE', ['BOOLEAN', 'DOUBLE', 'DOUBLE'], 'CUSTOM'],
        [['if'], 'DATETIME', ['BOOLEAN', 'DATETIME', 'DATETIME'], 'CUSTOM'],
        [['if'], 'DATE', ['BOOLEAN', 'DATE', 'DATE'], 'CUSTOM'],
        [['if'], 'DATETIMEV2', ['BOOLEAN', 'DATETIMEV2', 'DATETIMEV2'], 'CUSTOM'],
        [['if'], 'DATEV2', ['BOOLEAN', 'DATEV2', 'DATEV2'], 'CUSTOM'],
        [['if'], 'DECIMALV2', ['BOOLEAN', 'DECIMALV2', 'DECIMALV2'], 'CUSTOM'],
        [['if'], 'DECIMAL32', ['BOOLEAN', 'DECIMAL32', 'DECIMAL32'], 'CUSTOM'],
        [['if'], 'DECIMAL64', ['BOOLEAN', 'DECIMAL64', 'DECIMAL64'], 'CUSTOM'],
        [['if'], 'DECIMAL128', ['BOOLEAN', 'DECIMAL128', 'DECIMAL128'], 'CUSTOM'],
        [['if'], 'BITMAP', ['BOOLEAN', 'BITMAP', 'BITMAP'], 'CUSTOM'],
        [['if'], 'HLL', ['BOOLEAN', 'HLL', 'HLL'], 'CUSTOM'],
        [['if'], 'JSONB', ['BOOLEAN', 'JSONB', 'JSONB'], 'CUSTOM'],
        # The priority of varchar should be lower than decimal in IS_SUPERTYPE_OF mode.
        [['if'], 'VARCHAR', ['BOOLEAN', 'VARCHAR', 'VARCHAR'], 'CUSTOM'],
        [['if'], 'STRING', ['BOOLEAN', 'STRING', 'STRING'], 'CUSTOM'],
        [['if'], 'ARRAY_BOOLEAN', ['BOOLEAN', 'ARRAY_BOOLEAN', 'ARRAY_BOOLEAN'], 'CUSTOM'],
        [['if'], 'ARRAY_TINYINT', ['BOOLEAN', 'ARRAY_TINYINT', 'ARRAY_TINYINT'], 'CUSTOM'],
        [['if'], 'ARRAY_SMALLINT', ['BOOLEAN', 'ARRAY_SMALLINT', 'ARRAY_SMALLINT'], 'CUSTOM'],
        [['if'], 'ARRAY_INT', ['BOOLEAN', 'ARRAY_INT', 'ARRAY_INT'], 'CUSTOM'],
        [['if'], 'ARRAY_BIGINT', ['BOOLEAN', 'ARRAY_BIGINT', 'ARRAY_BIGINT'], 'CUSTOM'],
        [['if'], 'ARRAY_LARGEINT', ['BOOLEAN', 'ARRAY_LARGEINT', 'ARRAY_LARGEINT'], 'CUSTOM'],
        [['if'], 'ARRAY_FLOAT', ['BOOLEAN', 'ARRAY_FLOAT', 'ARRAY_FLOAT'], 'CUSTOM'],
        [['if'], 'ARRAY_DOUBLE', ['BOOLEAN', 'ARRAY_DOUBLE', 'ARRAY_DOUBLE'], 'CUSTOM'],
        [['if'], 'ARRAY_DATETIME', ['BOOLEAN', 'ARRAY_DATETIME', 'ARRAY_DATETIME'], 'CUSTOM'],
        [['if'], 'ARRAY_DATE', ['BOOLEAN', 'ARRAY_DATE', 'ARRAY_DATE'], 'CUSTOM'],
        [['if'], 'ARRAY_DATETIMEV2', ['BOOLEAN', 'ARRAY_DATETIMEV2', 'ARRAY_DATETIMEV2'], 'CUSTOM'],
        [['if'], 'ARRAY_DATEV2', ['BOOLEAN', 'ARRAY_DATEV2', 'ARRAY_DATEV2'], 'CUSTOM'],
        [['if'], 'ARRAY_DECIMALV2', ['BOOLEAN', 'ARRAY_DECIMALV2', 'ARRAY_DECIMALV2'], 'CUSTOM'],
        [['if'], 'ARRAY_VARCHAR', ['BOOLEAN', 'ARRAY_VARCHAR', 'ARRAY_VARCHAR'], 'CUSTOM'],
        [['if'], 'ARRAY_STRING', ['BOOLEAN', 'ARRAY_STRING', 'ARRAY_STRING'], 'CUSTOM'],

        [['nullif'], 'BOOLEAN', ['BOOLEAN', 'BOOLEAN'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'TINYINT', ['TINYINT', 'TINYINT'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'INT', ['INT', 'INT'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'BIGINT', ['BIGINT', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'LARGEINT', ['LARGEINT', 'LARGEINT'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'FLOAT', ['FLOAT', 'FLOAT'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'DATETIME', ['DATETIME', 'DATETIME'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'DATE', ['DATE', 'DATE'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'DATEV2', ['DATEV2', 'DATEV2'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'DECIMALV2', ['DECIMALV2', 'DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'DECIMAL32', ['DECIMAL32', 'DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'DECIMAL64', ['DECIMAL64', 'DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'DECIMAL128', ['DECIMAL128', 'DECIMAL128'], 'ALWAYS_NULLABLE'],
        # The priority of varchar should be lower than decimal in IS_SUPERTYPE_OF mode.
        [['nullif'], 'VARCHAR', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['nullif'], 'STRING', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],

        [['is_null_pred'], "BOOLEAN", ['T', '...'], 'ALWAYS_NULLABLE', ['T']],
        [['is_not_null_pred'], "BOOLEAN", ['T', '...'], 'ALWAYS_NULLABLE', ['T']],

        [['ifnull', 'nvl'], 'BOOLEAN', ['BOOLEAN', 'BOOLEAN'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'TINYINT', ['TINYINT', 'TINYINT'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'INT', ['INT', 'INT'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'BIGINT', ['BIGINT', 'BIGINT'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'LARGEINT', ['LARGEINT', 'LARGEINT'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'FLOAT', ['FLOAT', 'FLOAT'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DATE', ['DATE', 'DATE'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DATETIME', ['DATETIME', 'DATETIME'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DATETIME', ['DATE', 'DATETIME'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DATETIME', ['DATETIME', 'DATE'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DATEV2', ['DATEV2', 'DATEV2'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DATETIMEV2', ['DATETIMEV2', 'DATETIMEV2'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DECIMALV2', ['DECIMALV2', 'DECIMALV2'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DECIMAL32', ['DECIMAL32', 'DECIMAL32'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DECIMAL64', ['DECIMAL64', 'DECIMAL64'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'DECIMAL128', ['DECIMAL128', 'DECIMAL128'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'BITMAP', ['BITMAP', 'BITMAP'], 'CUSTOM'],
        # The priority of varchar should be lower than decimal in IS_SUPERTYPE_OF mode.
        [['ifnull', 'nvl'], 'VARCHAR', ['VARCHAR', 'VARCHAR'], 'CUSTOM'],
        [['ifnull', 'nvl'], 'STRING', ['STRING', 'STRING'], 'CUSTOM'],

        [['coalesce'], 'BOOLEAN', ['BOOLEAN', '...'], 'CUSTOM'],
        [['coalesce'], 'TINYINT', ['TINYINT', '...'], 'CUSTOM'],
        [['coalesce'], 'SMALLINT', ['SMALLINT', '...'], 'CUSTOM'],
        [['coalesce'], 'INT', ['INT', '...'], 'CUSTOM'],
        [['coalesce'], 'BIGINT', ['BIGINT', '...'], 'CUSTOM'],
        [['coalesce'], 'LARGEINT', ['LARGEINT', '...'], 'CUSTOM'],
        [['coalesce'], 'FLOAT', ['FLOAT', '...'], 'CUSTOM'],
        [['coalesce'], 'DOUBLE', ['DOUBLE', '...'], 'CUSTOM'],
        [['coalesce'], 'DATETIME', ['DATETIME', '...'], 'CUSTOM'],
        [['coalesce'], 'DATE', ['DATE', '...'], 'CUSTOM'],
        [['coalesce'], 'DATETIMEV2', ['DATETIMEV2', '...'], 'CUSTOM'],
        [['coalesce'], 'DATEV2', ['DATEV2', '...'], 'CUSTOM'],
        [['coalesce'], 'DECIMALV2', ['DECIMALV2', '...'], 'CUSTOM'],
        [['coalesce'], 'DECIMAL32', ['DECIMAL32', '...'], 'CUSTOM'],
        [['coalesce'], 'DECIMAL64', ['DECIMAL64', '...'], 'CUSTOM'],
        [['coalesce'], 'DECIMAL128', ['DECIMAL128', '...'], 'CUSTOM'],
        [['coalesce'], 'BITMAP', ['BITMAP', '...'], 'CUSTOM'],
        # The priority of varchar should be lower than decimal in IS_SUPERTYPE_OF mode.
        [['coalesce'], 'VARCHAR', ['VARCHAR', '...'], 'CUSTOM'],
        [['coalesce'], 'STRING', ['STRING', '...'], 'CUSTOM'],

        # esquery's first arg is suitable for all types
        [['esquery'], 'BOOLEAN', ['BOOLEAN', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['TINYINT', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['SMALLINT', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['INT', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['BIGINT', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['LARGEINT', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['FLOAT', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['DOUBLE', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['DATE', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['DATETIME', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['CHAR', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['VARCHAR', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['JSONB', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['DECIMALV2', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['DECIMAL32', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['DECIMAL64', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['DECIMAL128', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['TIMEV2', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['DATEV2', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['DATETIMEV2', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['TIMEV2', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['ARRAY<T>', 'VARCHAR'], '', ['T']],
        [['esquery'], 'BOOLEAN', ['MAP', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['STRING', 'VARCHAR'], ''],
        [['esquery'], 'BOOLEAN', ['VARIANT', 'VARCHAR'], ''],
        # used for accept graph sql
        [['g'], 'BOOLEAN', ['VARCHAR'], '']
    ],

    # String builtin functions
    "String": [
        [['decode_as_varchar'], 'VARCHAR', ['SMALLINT'], 'DEPEND_ON_ARGUMENT'],
        [['decode_as_varchar'], 'VARCHAR', ['INT'], 'DEPEND_ON_ARGUMENT'],
        [['decode_as_varchar'], 'VARCHAR', ['BIGINT'], 'DEPEND_ON_ARGUMENT'],
        [['decode_as_varchar'], 'VARCHAR', ['LARGEINT'], 'DEPEND_ON_ARGUMENT'],
        [['encode_as_smallint'], 'SMALLINT', ['VARCHAR'], 'DEPEND_ON_ARGUMENT'],
        [['encode_as_int'], 'INT', ['VARCHAR'], 'DEPEND_ON_ARGUMENT'],
        [['encode_as_bigint'], 'BIGINT', ['VARCHAR'], 'DEPEND_ON_ARGUMENT'],
        [['encode_as_largeint'], 'LARGEINT', ['VARCHAR'], 'DEPEND_ON_ARGUMENT'],
        [['substr', 'substring'], 'VARCHAR', ['VARCHAR', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['substr', 'substring'], 'VARCHAR', ['VARCHAR', 'INT', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['mask'], 'STRING', ['STRING', '...'], ''],
        [['mask_first_n'], 'STRING', ['STRING'], ''],
        [['mask_first_n'], 'STRING', ['STRING', 'INT'], ''],
        [['mask_last_n'], 'STRING', ['STRING'], ''],
        [['mask_last_n'], 'STRING', ['STRING', 'INT'], ''],
        [['strleft', 'left'], 'VARCHAR', ['VARCHAR', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['strright', 'right'], 'VARCHAR', ['VARCHAR', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['ends_with'], 'BOOLEAN', ['VARCHAR', 'VARCHAR'], ''],
        [['starts_with'], 'BOOLEAN', ['VARCHAR', 'VARCHAR'], ''],
        [['null_or_empty'], 'BOOLEAN', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['not_null_or_empty'], 'BOOLEAN', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['space'], 'VARCHAR', ['INT'], ''],
        [['repeat'], 'VARCHAR', ['VARCHAR', 'INT'], 'ALWAYS_NULLABLE'],
        [['lpad'], 'VARCHAR', ['VARCHAR', 'INT', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['rpad'], 'VARCHAR', ['VARCHAR', 'INT', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['append_trailing_char_if_absent'], 'VARCHAR', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['length'], 'INT', ['VARCHAR'], ''],
        [['crc32'], 'BIGINT', ['VARCHAR'], ''],
        [['bit_length'], 'INT', ['VARCHAR'], ''],

        [['char_length', 'character_length'], 'INT', ['VARCHAR'], ''],
        [['lower', 'lcase'], 'VARCHAR', ['VARCHAR'], ''],
        [['upper', 'ucase'], 'VARCHAR', ['VARCHAR'], ''],
        [['quote'], 'VARCHAR', ['VARCHAR'], ''],
        [['auto_partition_name'], 'VARCHAR', ['VARCHAR', '...'], ''],
        [['initcap'], 'VARCHAR', ['VARCHAR'], ''],
        [['trim'], 'VARCHAR', ['VARCHAR','VARCHAR'], ''],
        [['trim'], 'VARCHAR', ['VARCHAR'], ''],
        [['ltrim'], 'VARCHAR', ['VARCHAR'], ''],
        [['ltrim'], 'VARCHAR', ['VARCHAR','VARCHAR'], ''],
        [['rtrim'], 'VARCHAR', ['VARCHAR'], ''],
        [['rtrim'], 'VARCHAR', ['VARCHAR','VARCHAR'], ''],
        [['ascii'], 'INT', ['VARCHAR'], ''],
        [['instr'], 'INT', ['VARCHAR', 'VARCHAR'], ''],
        [['locate'], 'INT', ['VARCHAR', 'VARCHAR'], ''],
        [['locate'], 'INT', ['VARCHAR', 'VARCHAR', 'INT'], ''],
        [['regexp_extract'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['regexp_replace'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['regexp_replace_one'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['regexp_extract_all'], 'VARCHAR', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['concat'], 'VARCHAR', ['VARCHAR', '...'], ''],
        [['elt'], 'VARCHAR', ['INT', 'VARCHAR', '...'], 'ALWAYS_NULLABLE'],
        [['elt'], 'STRING', ['INT', 'STRING', '...'], 'ALWAYS_NULLABLE'],
        [['replace'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR'], ''],
        [['concat_ws'], 'VARCHAR', ['VARCHAR', 'VARCHAR', '...'], 'CUSTOM'],
        [['concat_ws'], 'VARCHAR', ['VARCHAR', 'ARRAY_VARCHAR'], 'CUSTOM'],
        [['find_in_set'], 'INT', ['VARCHAR', 'VARCHAR'], ''],
        [['parse_url'], 'VARCHAR', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['parse_url'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['money_format'], 'VARCHAR', ['BIGINT'], ''],
        [['money_format'], 'VARCHAR', ['LARGEINT'], ''],
        [['money_format'], 'VARCHAR', ['DOUBLE'], ''],
        [['money_format'], 'VARCHAR', ['DECIMALV2'], ''],
        [['money_format'], 'VARCHAR', ['DECIMAL32'], ''],
        [['money_format'], 'VARCHAR', ['DECIMAL64'], ''],
        [['money_format'], 'VARCHAR', ['DECIMAL128'], ''],
        [['split_by_string'],'ARRAY_VARCHAR',['STRING','STRING'], ''],
        [['split_by_regexp'],'ARRAY_VARCHAR',['STRING','STRING'], ''],
        [['split_by_regexp'],'ARRAY_VARCHAR',['STRING','STRING', 'INT'], ''],
        [['split_part'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'INT'], 'ALWAYS_NULLABLE'],
        [['substring_index'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['extract_url_parameter'], 'VARCHAR', ['VARCHAR', 'VARCHAR'], ''],
        [['url_decode'], 'VARCHAR', ['VARCHAR'], ''],
        [['random_bytes'], 'VARCHAR', ['INT'], ''],

        [['sub_replace'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'INT'], 'ALWAYS_NULLABLE'],
        [['sub_replace'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'INT', 'INT'], 'ALWAYS_NULLABLE'],

        [['char'], 'VARCHAR', ['VARCHAR', 'INT', '...'], 'ALWAYS_NULLABLE'],
        [['strcmp'], 'INT', ['VARCHAR', 'VARCHAR'], 'DEPEND_ON_ARGUMENT'],

        [['overlay'], 'VARCHAR', ['VARCHAR', 'INT', 'INT', 'VARCHAR'], ''],

        [['count_substrings'], 'INT', ['STRING', 'STRING'], 'DEPEND_ON_ARGUMENT'],
        [['substr', 'substring'], 'STRING', ['STRING', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['substr', 'substring'], 'STRING', ['STRING', 'INT', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['strleft', 'left'], 'STRING', ['STRING', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['strright', 'right'], 'STRING', ['STRING', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['ends_with'], 'BOOLEAN', ['STRING', 'STRING'], ''],
        [['starts_with'], 'BOOLEAN', ['STRING', 'STRING'], ''],
        [['null_or_empty'], 'BOOLEAN', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['not_null_or_empty'], 'BOOLEAN', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['space'], 'STRING', ['INT'], ''],
        [['repeat'], 'STRING', ['STRING', 'INT'], 'ALWAYS_NULLABLE'],
        [['lpad'], 'STRING', ['STRING', 'INT', 'STRING'], 'ALWAYS_NULLABLE'],
        [['rpad'], 'STRING', ['STRING', 'INT', 'STRING'], 'ALWAYS_NULLABLE'],
        [['format_round'], 'STRING', ['BIGINT'], ''],
        [['format_round'], 'STRING', ['LARGEINT'], ''],
        [['format_round'], 'STRING', ['DOUBLE', 'INT'], ''],
        [['format_round'], 'STRING', ['DECIMAL32', 'INT'], ''],
        [['format_round'], 'STRING', ['DECIMAL64', 'INT'], ''],
        [['format_round'], 'STRING', ['DECIMAL128', 'INT'], ''],
        [['append_trailing_char_if_absent'], 'STRING', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['length'], 'INT', ['STRING'], ''],
        [['crc32'], 'BIGINT', ['STRING'], ''],
        [['bit_length'], 'INT', ['STRING'], ''],

        [['char_length', 'character_length'], 'INT', ['STRING'], ''],
        [['lower', 'lcase'], 'STRING', ['STRING'], ''],
        [['upper', 'ucase'], 'STRING', ['STRING'], ''],
        [['quote'], 'STRING', ['STRING'], ''],
        [['trim'], 'STRING', ['STRING'], ''],
        [['trim'], 'STRING', ['STRING','STRING'], ''],
        [['ltrim'], 'STRING', ['STRING'], ''],
        [['ltrim'], 'STRING', ['STRING','STRING'], ''],
        [['rtrim'], 'STRING', ['STRING'], ''],
        [['rtrim'], 'STRING', ['STRING','STRING'], ''],
        [['auto_partition_name'], 'STRING', ['STRING', '...'], ''],
        [['ascii'], 'INT', ['STRING'], ''],
        [['instr'], 'INT', ['STRING', 'STRING'], ''],
        [['locate'], 'INT', ['STRING', 'STRING'], ''],
        [['locate'], 'INT', ['STRING', 'STRING', 'INT'], ''],
        [['regexp_extract'], 'STRING', ['STRING', 'STRING', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['regexp_replace'], 'STRING', ['STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['regexp_replace_one'], 'STRING', ['STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['regexp_extract_all'], 'STRING', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['concat'], 'STRING', ['STRING', '...'], ''],
        [['replace'], 'STRING', ['STRING', 'STRING', 'STRING'], ''],
        [['concat_ws'], 'STRING', ['STRING', 'STRING', '...'], 'CUSTOM'],
        [['find_in_set'], 'INT', ['STRING', 'STRING'], ''],
        [['parse_url'], 'STRING', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['parse_url'], 'STRING', ['STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['money_format'], 'STRING', ['BIGINT'], ''],
        [['money_format'], 'STRING', ['LARGEINT'], ''],
        [['money_format'], 'STRING', ['DOUBLE'], ''],
        [['money_format'], 'STRING', ['DECIMALV2'], ''],
        [['money_format'], 'STRING', ['DECIMAL32'], ''],
        [['money_format'], 'STRING', ['DECIMAL64'], ''],
        [['money_format'], 'STRING', ['DECIMAL128'], ''],
        [['split_part'], 'STRING', ['STRING', 'STRING', 'INT'], 'ALWAYS_NULLABLE'],
        [['substring_index'], 'STRING', ['STRING', 'STRING', 'INT'], 'DEPEND_ON_ARGUMENT'],
        [['url_decode'], 'STRING', ['STRING'], ''],
        [['random_bytes'], 'STRING', ['INT'], ''],

        [['format'], 'STRING', ['STRING', 'TINYINT', '...'], ''],
        [['format'], 'STRING', ['STRING', 'SMALLINT', '...'], ''],
        [['format'], 'STRING', ['STRING', 'INT', '...'], ''],
        [['format'], 'STRING', ['STRING', 'BIGINT', '...'], ''],
        [['format'], 'STRING', ['STRING', 'LARGEINT', '...'], ''],
        [['format'], 'STRING', ['STRING', 'FLOAT', '...'], ''],
        [['format'], 'STRING', ['STRING', 'DOUBLE', '...'], ''],
        [['format'], 'STRING', ['STRING', 'STRING', '...'], ''],
        [['format_number'], 'STRING', ['DOUBLE'], ''],
        [['parse_data_size'], 'LARGEINT', ['STRING'], ''],
        [['overlay'], 'STRING', ['STRING', 'INT', 'INT', 'STRING'], ''],
        [['strcmp'], 'INT', ['STRING', 'STRING'], 'DEPEND_ON_ARGUMENT']
    ],


    # width_bucket
    "Width_Bucket": [
        [['width_bucket'], 'BIGINT', ['TINYINT','TINYINT','TINYINT','TINYINT'], ''],
        [['width_bucket'], 'BIGINT', ['SMALLINT','SMALLINT','SMALLINT','SMALLINT'], ''],
        [['width_bucket'], 'BIGINT', ['INT','INT','INT','INT'], ''],
        [['width_bucket'], 'BIGINT', ['BIGINT','BIGINT','BIGINT','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['FLOAT','FLOAT','FLOAT','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['DOUBLE','DOUBLE','DOUBLE','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['DECIMAL32','DECIMAL32','DECIMAL32','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['DECIMAL64','DECIMAL64','DECIMAL64','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['DECIMAL128','DECIMAL128','DECIMAL128','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['DECIMALV2','DECIMALV2','DECIMALV2','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['DATE','DATE','DATE','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['DATEV2','DATEV2','DATEV2','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['DATETIME','DATETIME','DATETIME','BIGINT'], ''],
        [['width_bucket'], 'BIGINT', ['DATETIMEV2','DATETIMEV2','DATETIMEV2','BIGINT'], '']
    ],

    # Utility functions
    "Utility": [
        [['convert_to'], 'VARCHAR', ['VARCHAR','VARCHAR'], ''],
        [['sleep'], 'BOOLEAN', ['INT'], ''],
        [['version'], 'VARCHAR', [], 'ALWAYS_NOT_NULLABLE'],
        [['field'], 'INT', ['TINYINT', '...'], ''],
        [['field'], 'INT', ['SMALLINT', '...'], ''],
        [['field'], 'INT', ['INT', '...'], ''],
        [['field'], 'INT', ['BIGINT', '...'], ''],
        [['field'], 'INT', ['LARGEINT', '...'], ''],
        [['field'], 'INT', ['FLOAT', '...'], ''],
        [['field'], 'INT', ['DOUBLE', '...'], ''],
        [['field'], 'INT', ['DECIMALV2', '...'], ''],
        [['field'], 'INT', ['DATEV2', '...'], ''],
        [['field'], 'INT', ['DATETIMEV2', '...'], ''],
        [['field'], 'INT', ['VARCHAR', '...'], ''],
        [['field'], 'INT', ['STRING', '...'], '']
    ],


    # JSON functions
    "Json": [
        [['jsonb_parse'], 'JSONB', ['VARCHAR'], ''],
        [['jsonb_parse_error_to_null'], 'JSONB', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_parse_error_to_value'], 'JSONB', ['VARCHAR', 'VARCHAR'], ''],
        [['jsonb_parse_error_to_invalid'], 'JSONB', ['VARCHAR'], ''],
        [['jsonb_parse_nullable'], 'JSONB', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_parse_nullable_error_to_null'], 'JSONB', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_parse_nullable_error_to_value'], 'JSONB', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_parse_nullable_error_to_invalid'], 'JSONB', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_parse_notnull'], 'JSONB', ['VARCHAR'], ''],
        [['jsonb_parse_notnull_error_to_value'], 'JSONB', ['VARCHAR', 'VARCHAR'], ''],
        [['jsonb_parse_notnull_error_to_invalid'], 'JSONB', ['VARCHAR'], ''],

        [['jsonb_exists_path'], 'BOOLEAN', ['JSONB', 'VARCHAR'], ''],
        [['jsonb_exists_path'], 'BOOLEAN', ['JSONB', 'STRING'], ''],
        [['jsonb_type'], 'STRING', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_type'], 'STRING', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],

        [['jsonb_keys'], 'ARRAY_STRING', ['JSONB'], 'ALWAYS_NULLABLE'],
        [['jsonb_keys'], 'ARRAY_STRING', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['json_keys'], 'ARRAY_STRING', ['JSONB'], 'ALWAYS_NULLABLE'],
        [['json_keys'], 'ARRAY_STRING', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],

        [['jsonb_extract'], 'JSONB', ['JSONB', 'VARCHAR', '...'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract'], 'JSONB', ['JSONB', 'STRING', '...'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_isnull'], 'BOOLEAN', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_isnull'], 'BOOLEAN', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_bool'], 'BOOLEAN', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_bool'], 'BOOLEAN', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_int'], 'INT', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_int'], 'INT', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_bigint'], 'BIGINT', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_bigint'], 'BIGINT', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_largeint'], 'LARGEINT', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_largeint'], 'LARGEINT', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_double'], 'DOUBLE', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_double'], 'DOUBLE', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_string'], 'STRING', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['jsonb_extract_string'], 'STRING', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],

        [['json_parse'], 'JSONB', ['VARCHAR'], ''],
        [['json_parse_error_to_null'], 'JSONB', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_parse_error_to_value'], 'JSONB', ['VARCHAR', 'VARCHAR'], ''],
        [['json_parse_error_to_invalid'], 'JSONB', ['VARCHAR'], ''],
        [['json_parse_nullable'], 'JSONB', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_parse_nullable_error_to_null'], 'JSONB', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_parse_nullable_error_to_value'], 'JSONB', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_parse_nullable_error_to_invalid'], 'JSONB', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_parse_notnull'], 'JSONB', ['VARCHAR'], ''],
        [['json_parse_notnull_error_to_value'], 'JSONB', ['VARCHAR', 'VARCHAR'], ''],
        [['json_parse_notnull_error_to_invalid'], 'JSONB', ['VARCHAR'], ''],

        [['json_search'], 'JSONB', ['VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],

        [['json_exists_path'], 'BOOLEAN', ['JSONB', 'VARCHAR'], ''],
        [['json_exists_path'], 'BOOLEAN', ['JSONB', 'STRING'], ''],
        [['json_type'], 'STRING', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_type'], 'STRING', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],

        [['json_extract_isnull'], 'BOOLEAN', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_extract_isnull'], 'BOOLEAN', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['json_extract_bool'], 'BOOLEAN', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_extract_bool'], 'BOOLEAN', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['json_extract_int'], 'INT', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_extract_int'], 'INT', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['json_extract_bigint'], 'BIGINT', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_extract_bigint'], 'BIGINT', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['json_extract_largeint'], 'LARGEINT', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_extract_largeint'], 'LARGEINT', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['json_extract_double'], 'DOUBLE', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_extract_double'], 'DOUBLE', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['json_extract_string'], 'STRING', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_extract_string'], 'STRING', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['json_length'], 'INT', ['JSONB'], 'ALWAYS_NULLABLE'],
        [['json_length'], 'INT', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_length'], 'INT', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['json_contains'], 'BOOLEAN', ['JSONB', 'JSONB'], 'ALWAYS_NULLABLE'],
        [['json_contains'], 'BOOLEAN', ['JSONB', 'JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_contains'], 'BOOLEAN', ['JSONB', 'JSONB', 'STRING'], 'ALWAYS_NULLABLE'],

        # Json functions
        [['get_json_int'], 'INT', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['get_json_double'], 'DOUBLE', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['get_json_string'], 'VARCHAR', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['get_json_int'], 'INT', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['get_json_double'], 'DOUBLE', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['get_json_string'], 'STRING', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['get_json_bigint'], 'BIGINT', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['get_json_bigint'], 'BIGINT', ['STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['get_json_string'], 'STRING', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['get_json_string'], 'STRING', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['get_json_int'], 'INT', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['get_json_int'], 'INT', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['get_json_double'], 'DOUBLE', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['get_json_double'], 'DOUBLE', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],
        [['get_json_bigint'], 'BIGINT', ['JSONB', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['get_json_bigint'], 'BIGINT', ['JSONB', 'STRING'], 'ALWAYS_NULLABLE'],

        [['json_array'], 'VARCHAR', ['VARCHAR', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['json_object'], 'VARCHAR', ['VARCHAR', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['json_quote'], 'VARCHAR', ['VARCHAR'], ''],
        [['json_valid'], 'INT', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_contains'], 'BOOLEAN', ['VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_unquote'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['json_extract'], 'VARCHAR', ['VARCHAR', 'VARCHAR', '...'], 'ALWAYS_NULLABLE'],
        [['json_extract'], 'STRING', ['STRING', 'STRING', '...'], 'ALWAYS_NULLABLE'],
        [['json_insert'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR', '...'], ''],
        [['json_replace'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR', '...'], ''],
        [['json_set'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR', '...'], '']
    ],


    #hll function
    "Hll": [
        [['hll_cardinality'], 'BIGINT', ['HLL'], 'ALWAYS_NOT_NULLABLE'],
        [['hll_empty'], 'HLL', [], 'ALWAYS_NOT_NULLABLE'],
        [['hll_from_base64'], 'HLL', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['hll_from_base64'], 'HLL', ['STRING'], 'ALWAYS_NULLABLE'],
        [['hll_hash'], 'HLL', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['hll_hash'], 'HLL', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['hll_to_base64'], 'STRING', ['HLL'], '']
    ],


    #bitmap function
    "Bitmap": [
        [['to_bitmap'], 'BITMAP', ['BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['to_bitmap'], 'BITMAP', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['to_bitmap'], 'BITMAP', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['to_bitmap_with_check'], 'BITMAP', ['BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['to_bitmap_with_check'], 'BITMAP', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['to_bitmap_with_check'], 'BITMAP', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['bitmap_hash'], 'BITMAP', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['bitmap_hash64'], 'BITMAP', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['bitmap_hash'], 'BITMAP', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['bitmap_hash64'], 'BITMAP', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['bitmap_count'], 'BIGINT', ['BITMAP'], 'ALWAYS_NOT_NULLABLE'],
        [['bitmap_and_not_count', 'bitmap_andnot_count'], 'BIGINT', ['BITMAP','BITMAP'], ''],
        [['bitmap_empty'], 'BITMAP', [], 'ALWAYS_NOT_NULLABLE'],
        [['bitmap_or'], 'BITMAP', ['BITMAP','BITMAP','...'], ''],
        [['bitmap_or'], 'BITMAP', ['BITMAP','BITMAP'], ''],
        [['bitmap_xor'], 'BITMAP', ['BITMAP','BITMAP','...'], ''],
        [['bitmap_xor'], 'BITMAP', ['BITMAP','BITMAP'], ''],
        [['bitmap_xor_count'], 'BIGINT', ['BITMAP','BITMAP','...'], ''],
        [['bitmap_xor_count'], 'BIGINT', ['BITMAP','BITMAP'], ''],
        [['bitmap_not'], 'BITMAP', ['BITMAP','BITMAP'], ''],
        [['bitmap_and'], 'BITMAP', ['BITMAP','BITMAP','...'], ''],
        [['bitmap_and'], 'BITMAP', ['BITMAP','BITMAP'], ''],
        [['bitmap_and_not', 'bitmap_andnot'], 'BITMAP', ['BITMAP','BITMAP'], ''],
        [['bitmap_to_string'], 'STRING', ['BITMAP'], ''],
        [['bitmap_from_string'], 'BITMAP', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['bitmap_from_string'], 'BITMAP', ['STRING'], 'ALWAYS_NULLABLE'],
        [['bitmap_to_base64'], 'STRING', ['BITMAP'], ''],
        [['bitmap_from_base64'], 'BITMAP', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['bitmap_from_base64'], 'BITMAP', ['STRING'], 'ALWAYS_NULLABLE'],
        [['bitmap_from_array'], 'BITMAP', ['ARRAY_TINYINT'], 'ALWAYS_NULLABLE'],
        [['bitmap_from_array'], 'BITMAP', ['ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['bitmap_from_array'], 'BITMAP', ['ARRAY_INT'], 'ALWAYS_NULLABLE'],
        [['bitmap_from_array'], 'BITMAP', ['ARRAY_BIGINT'], 'ALWAYS_NULLABLE'],
        [['bitmap_contains'], 'BOOLEAN', ['BITMAP','BIGINT'], ''],
        [['bitmap_remove'], 'BITMAP', ['BITMAP','BIGINT'], ''],
        [['bitmap_has_any'], 'BOOLEAN', ['BITMAP','BITMAP'], ''],
        [['bitmap_has_all'], 'BOOLEAN', ['BITMAP','BITMAP'], ''],
        [['bitmap_min'], 'BIGINT', ['BITMAP'], 'ALWAYS_NULLABLE'],
        [['bitmap_max'], 'BIGINT', ['BITMAP'], 'ALWAYS_NULLABLE'],
        [['bitmap_subset_in_range'], 'BITMAP', ['BITMAP', 'BIGINT', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['bitmap_subset_limit'], 'BITMAP', ['BITMAP', 'BIGINT', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['bitmap_and_count'], 'BIGINT', ['BITMAP','BITMAP','...'], ''],
        [['bitmap_and_count'], 'BIGINT', ['BITMAP','BITMAP'], ''],
        [['bitmap_or_count'], 'BIGINT', ['BITMAP','BITMAP','...'], ''],
        [['bitmap_or_count'], 'BIGINT', ['BITMAP','BITMAP'], ''],
        [['sub_bitmap'], 'BITMAP', ['BITMAP', 'BIGINT', 'BIGINT'], 'ALWAYS_NULLABLE'],
        [['bitmap_to_array'], 'ARRAY_BIGINT', ['BITMAP'], '']
    ],


    # quantile_function
    "quantile" : [
        [['quantile_state_empty'], 'QUANTILE_STATE', [], 'ALWAYS_NOT_NULLABLE'],
        [['to_quantile_state'], 'QUANTILE_STATE', ['DOUBLE', 'FLOAT'], 'ALWAYS_NOT_NULLABLE'],
        [['quantile_percent'], 'DOUBLE', ['QUANTILE_STATE', 'FLOAT'], 'ALWAYS_NOT_NULLABLE']
    ],

    # hash functions
    "Hash": [
        [['murmur_hash3_32'], 'INT', ['VARCHAR', '...'], ''],
        [['murmur_hash3_32'], 'INT', ['STRING', '...'], ''],
        [['murmur_hash3_64'], 'BIGINT', ['VARCHAR', '...'], ''],
        [['murmur_hash3_64'], 'BIGINT', ['STRING', '...'], ''],
        [['xxhash_32'], 'INT', ['VARCHAR', '...'], ''],
        [['xxhash_32'], 'INT', ['STRING', '...'], ''],
        [['xxhash_64'], 'BIGINT', ['VARCHAR', '...'], ''],
        [['xxhash_64'], 'BIGINT', ['STRING', '...'], '']
    ],

    # aes and base64 function
    "aes": [
        [['aes_encrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['aes_decrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['aes_encrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['aes_decrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['aes_encrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['aes_decrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['sm4_encrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['sm4_decrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['sm4_encrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['sm4_decrypt'], 'VARCHAR', ['VARCHAR', 'VARCHAR', 'VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE'],
        [['from_base64'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['aes_encrypt'], 'STRING', ['STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['aes_decrypt'], 'STRING', ['STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['aes_encrypt'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['aes_decrypt'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['aes_encrypt'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['aes_decrypt'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['sm4_encrypt'], 'STRING', ['STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['sm4_decrypt'], 'STRING', ['STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['sm4_encrypt'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['sm4_decrypt'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING'], 'ALWAYS_NULLABLE'],
        [['from_base64'], 'STRING', ['STRING'], 'ALWAYS_NULLABLE'],
        [['to_base64'], 'STRING', ['STRING'], 'DEPEND_ON_ARGUMENT'],
        [['to_base64'], 'VARCHAR', ['VARCHAR'], 'DEPEND_ON_ARGUMENT']
    ],

    # for compatable with MySQL
    "Compatable": [
        [['md5'], 'VARCHAR', ['VARCHAR'], ''],
        [['md5sum'], 'VARCHAR', ['VARCHAR', '...'], ''],
        [['md5'], 'VARCHAR', ['STRING'], ''],
        [['md5sum'], 'VARCHAR', ['STRING', '...'], ''],

        [['sm3'], 'VARCHAR', ['VARCHAR'], ''],
        [['sm3sum'], 'VARCHAR', ['VARCHAR', '...'], ''],
        [['sm3'], 'VARCHAR', ['STRING'], ''],
        [['sm3sum'], 'VARCHAR', ['STRING', '...'], ''],

        [['sha'], 'VARCHAR', ['VARCHAR'], ''],
        [['sha'], 'VARCHAR', ['STRING'], ''],
        [['sha1'], 'VARCHAR', ['VARCHAR'], ''],
        [['sha1'], 'VARCHAR', ['STRING'], ''],
        [['sha2'], 'VARCHAR', ['VARCHAR', 'INT'], ''],
        [['sha2'], 'VARCHAR', ['STRING', 'INT'], ''],
    ],

    # geo functions
    "Geo": [
        [['ST_Point'], 'VARCHAR', ['DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],
        [['ST_X'], 'DOUBLE', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_Y'], 'DOUBLE', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_X'], 'DOUBLE', ['STRING'], 'ALWAYS_NULLABLE'],
        [['ST_Y'], 'DOUBLE', ['STRING'], 'ALWAYS_NULLABLE'],

        [['ST_Distance_Sphere'], 'DOUBLE', ['DOUBLE', 'DOUBLE', 'DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],
        [['ST_Angle_Sphere'], 'DOUBLE', ['DOUBLE', 'DOUBLE', 'DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],

        [['ST_Angle'], 'DOUBLE', ['VARCHAR','VARCHAR','VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_Angle'], 'DOUBLE', ['STRING','STRING','STRING'], 'ALWAYS_NULLABLE'],
        [['ST_Azimuth'], 'DOUBLE', ['VARCHAR','VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_Azimuth'], 'DOUBLE', ['STRING','STRING'], 'ALWAYS_NULLABLE'],

        [['ST_Area_Square_Meters'], 'DOUBLE', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_Area_Square_Meters'], 'DOUBLE', ['STRING'], 'ALWAYS_NULLABLE'],
        [['ST_Area_Square_Km'], 'DOUBLE', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_Area_Square_Km'], 'DOUBLE', ['STRING'], 'ALWAYS_NULLABLE'],

        [['ST_AsText', 'ST_AsWKT'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_AsText', 'ST_AsWKT'], 'VARCHAR', ['STRING'], 'ALWAYS_NULLABLE'],
        [['ST_GeometryFromText', 'ST_GeomFromText'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_GeometryFromText', 'ST_GeomFromText'], 'VARCHAR', ['STRING'], 'ALWAYS_NULLABLE'],

        [['ST_GeometryFromWkb', 'ST_GeomFromWkb'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_GeometryFromWkb', 'ST_GeomFromWkb'], 'VARCHAR', ['STRING'], 'ALWAYS_NULLABLE'],

        [['ST_AsBinary'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_AsBinary'], 'VARCHAR', ['STRING'], 'ALWAYS_NULLABLE'],

        [['ST_LineFromText', 'ST_LineStringFromText'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_LineFromText', 'ST_LineStringFromText'], 'VARCHAR', ['STRING'], 'ALWAYS_NULLABLE'],

        [['ST_Polygon', 'ST_PolyFromText', 'ST_PolygonFromText'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ST_Polygon', 'ST_PolyFromText', 'ST_PolygonFromText'], 'VARCHAR', ['STRING'], 'ALWAYS_NULLABLE'],

        [['ST_Circle'], 'VARCHAR', ['DOUBLE', 'DOUBLE', 'DOUBLE'], 'ALWAYS_NULLABLE'],

        [['ST_Contains'], 'BOOLEAN', ['VARCHAR', 'VARCHAR'], 'ALWAYS_NULLABLE']
    ],

    # grouping sets functions
    "grouping": [
        [['grouping_id'], 'BIGINT', ['BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['grouping'], 'BIGINT', ['BIGINT'], 'ALWAYS_NOT_NULLABLE']
    ],


    # url functions
    "Url": [
        [['domain'], 'STRING', ['STRING'], ''],
        [['domain_without_www'], 'STRING', ['STRING'], ''],
        [['protocol'], 'STRING', ['STRING'], ''],
        [['top_level_domain'], 'STRING', ['STRING'], ''],
        [['cut_to_first_significant_subdomain'], 'STRING', ['STRING'], ''],
        [['first_significant_subdomain'], 'STRING', ['STRING'], '']
    ],

    # search functions
    "Search": [
        [['multi_search_all_positions'], 'ARRAY_INT', ['STRING', 'ARRAY_STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['multi_match_any'], 'TINYINT', ['STRING', 'ARRAY_STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['like'], 'BOOLEAN', ['VARCHAR', 'VARCHAR'], ''],
        [['rlike'], 'BOOLEAN', ['VARCHAR', 'VARCHAR'], ''],
        [['regexp'], 'BOOLEAN', ['VARCHAR', 'VARCHAR'], '']
    ],
    # tokenizer functions
    "Tokenize": [
        [['tokenize'],'STRING',['STRING','STRING'], ''],
    ],

    "UUID": [
        [['uuid'], 'VARCHAR', [], 'ALWAYS_NOT_NULLABLE'],
        [['uuid_numeric'], 'LARGEINT', [], 'ALWAYS_NOT_NULLABLE'],
        [['uuid_to_int'], 'LARGEINT', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['uuid_to_int'], 'LARGEINT', ['STRING'], 'ALWAYS_NULLABLE'],
        [['int_to_uuid'], 'VARCHAR', ['LARGEINT'], 'DEPEND_ON_ARGUMENT']
    ],

    #ip functions
    "IP": [
        [['ipv4_num_to_string','inet_ntoa'], 'VARCHAR', ['TINYINT'], 'ALWAYS_NULLABLE'],
        [['ipv4_num_to_string','inet_ntoa'], 'VARCHAR', ['SMALLINT'], 'ALWAYS_NULLABLE'],
        [['ipv4_num_to_string','inet_ntoa'], 'VARCHAR', ['INT'], 'ALWAYS_NULLABLE'],
        [['ipv4_num_to_string','inet_ntoa'], 'VARCHAR', ['BIGINT'], 'ALWAYS_NULLABLE'],
        [['ipv4_string_to_num'], 'BIGINT', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['ipv4_string_to_num'], 'BIGINT', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['ipv4_string_to_num_or_default'], 'BIGINT', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['ipv4_string_to_num_or_default'], 'BIGINT', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['ipv4_string_to_num_or_null','inet_aton'], 'BIGINT', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ipv4_string_to_num_or_null','inet_aton'], 'BIGINT', ['STRING'], 'ALWAYS_NULLABLE'],  
        [['ipv6_num_to_string','inet6_ntoa'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],     
        [['ipv6_num_to_string','inet6_ntoa'], 'STRING', ['STRING'], 'ALWAYS_NULLABLE'],
        [['ipv6_num_to_string','inet6_ntoa'], 'VARCHAR', ['IPV6'], 'ALWAYS_NULLABLE'],
        [['ipv6_string_to_num'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['ipv6_string_to_num'], 'STRING', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['ipv6_string_to_num_or_default'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['ipv6_string_to_num_or_default'], 'STRING', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['ipv6_string_to_num_or_null','inet6_aton'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ipv6_string_to_num_or_null','inet6_aton'], 'STRING', ['STRING'], 'ALWAYS_NULLABLE'],  
        [['is_ipv4_compat'], 'BOOLEAN', ['VARCHAR'], ''],
        [['is_ipv4_compat'], 'BOOLEAN', ['STRING'], ''],
        [['is_ipv4_mapped'], 'BOOLEAN', ['VARCHAR'], ''],
        [['is_ipv4_mapped'], 'BOOLEAN', ['STRING'], ''],
        [['is_ipv4_string'], 'BOOLEAN', ['VARCHAR'], ''],
        [['is_ipv4_string'], 'BOOLEAN', ['STRING'], ''],
        [['is_ipv6_string'], 'BOOLEAN', ['VARCHAR'], ''],
        [['is_ipv6_string'], 'BOOLEAN', ['STRING'], ''],
        [['is_ip_address_in_range'], 'BOOLEAN', ['VARCHAR', 'VARCHAR'], 'DEPEND_ON_ARGUMENT'],
        [['is_ip_address_in_range'], 'BOOLEAN', ['STRING', 'STRING'], 'DEPEND_ON_ARGUMENT'],
        [['ipv4_cidr_to_range'], 'STRUCT<IPV4, IPV4>', ['IPV4', 'SMALLINT'], 'DEPEND_ON_ARGUMENT'],
        [['ipv6_cidr_to_range'], 'STRUCT<IPV6, IPV6>', ['IPV6', 'SMALLINT'], 'DEPEND_ON_ARGUMENT'],
        [['ipv6_cidr_to_range'], 'STRUCT<IPV6, IPV6>', ['VARCHAR', 'SMALLINT'], 'DEPEND_ON_ARGUMENT'],
        [['ipv6_cidr_to_range'], 'STRUCT<IPV6, IPV6>', ['STRING', 'SMALLINT'], 'DEPEND_ON_ARGUMENT'],
        [['to_ipv4'], 'IPV4', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['to_ipv4'], 'IPV4', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['to_ipv4_or_default'], 'IPV4', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['to_ipv4_or_default'], 'IPV4', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['to_ipv4_or_null'], 'IPV4', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['to_ipv4_or_null'], 'IPV4', ['STRING'], 'ALWAYS_NULLABLE'],
        [['to_ipv6'], 'IPV6', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['to_ipv6'], 'IPV6', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['to_ipv6_or_default'], 'IPV6', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['to_ipv6_or_default'], 'IPV6', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['to_ipv6_or_null'], 'IPV6', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['to_ipv6_or_null'], 'IPV6', ['STRING'], 'ALWAYS_NULLABLE'],
        [['ipv4_to_ipv6'], 'IPV6', ['IPV4'], 'DEPEND_ON_ARGUMENT'],
        [['cut_ipv6'], 'STRING', ['IPV6', 'TINYINT', 'TINYINT'], 'DEPEND_ON_ARGUMENT'],
        [['ipv6_from_uint128_string_or_null'], 'IPV6', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['ipv6_from_uint128_string_or_null'], 'IPV6', ['STRING'], 'ALWAYS_NULLABLE']
    ],

    "NonNullalbe": [
        [['non_nullable'], 'BOOLEAN', ['BOOLEAN'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'TINYINT', ['TINYINT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'SMALLINT', ['SMALLINT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'INT', ['INT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'BIGINT', ['BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'LARGEINT', ['LARGEINT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'FLOAT', ['FLOAT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'DOUBLE', ['DOUBLE'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'DATE', ['DATE'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'DATEV2', ['DATEV2'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'DATETIME', ['DATETIME'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'DECIMALV2', ['DECIMALV2'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'DECIMAL32', ['DECIMAL32'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'DECIMAL64', ['DECIMAL64'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'DECIMAL128', ['DECIMAL128'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'STRING', ['STRING'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'BITMAP', ['BITMAP'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'JSONB', ['JSONB'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_BOOLEAN', ['ARRAY_BOOLEAN'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_TINYINT', ['ARRAY_TINYINT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_INT', ['ARRAY_INT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_BIGINT', ['ARRAY_BIGINT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_DATETIME', ['ARRAY_DATETIME'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_DATE', ['ARRAY_DATE'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_DATETIMEV2', ['ARRAY_DATETIMEV2'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_DATEV2', ['ARRAY_DATEV2'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_FLOAT', ['ARRAY_FLOAT'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_DOUBLE', ['ARRAY_DOUBLE'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_VARCHAR', ['ARRAY_VARCHAR'], 'ALWAYS_NOT_NULLABLE'],
        [['non_nullable'], 'ARRAY_STRING', ['ARRAY_STRING'], 'ALWAYS_NOT_NULLABLE']
    ],

    "Nullalbe": [
        [['nullable'], 'BOOLEAN', ['BOOLEAN'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'TINYINT', ['TINYINT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'SMALLINT', ['SMALLINT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'INT', ['INT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'BIGINT', ['BIGINT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'LARGEINT', ['LARGEINT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'FLOAT', ['FLOAT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'DOUBLE', ['DOUBLE'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'DATE', ['DATE'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'DATEV2', ['DATEV2'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'DATETIME', ['DATETIME'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'DATETIMEV2', ['DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'DECIMALV2', ['DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'DECIMAL32', ['DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'DECIMAL64', ['DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'DECIMAL128', ['DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'VARCHAR', ['VARCHAR'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'STRING', ['STRING'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'BITMAP', ['BITMAP'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'JSONB', ['JSONB'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_BOOLEAN', ['ARRAY_BOOLEAN'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_TINYINT', ['ARRAY_TINYINT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_SMALLINT', ['ARRAY_SMALLINT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_INT', ['ARRAY_INT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_BIGINT', ['ARRAY_BIGINT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_LARGEINT', ['ARRAY_LARGEINT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_DATETIME', ['ARRAY_DATETIME'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_DATE', ['ARRAY_DATE'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_DATETIMEV2', ['ARRAY_DATETIMEV2'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_DATEV2', ['ARRAY_DATEV2'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_FLOAT', ['ARRAY_FLOAT'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_DOUBLE', ['ARRAY_DOUBLE'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_DECIMALV2', ['ARRAY_DECIMALV2'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_DECIMAL32', ['ARRAY_DECIMAL32'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_DECIMAL64', ['ARRAY_DECIMAL64'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_DECIMAL128', ['ARRAY_DECIMAL128'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_VARCHAR', ['ARRAY_VARCHAR'], 'ALWAYS_NULLABLE'],
        [['nullable'], 'ARRAY_STRING', ['ARRAY_STRING'], 'ALWAYS_NULLABLE']
    ],

    "Ignore": [
        [['ignore'], 'BOOLEAN', ['BOOLEAN', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['TINYINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['SMALLINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['INT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['BIGINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['LARGEINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['FLOAT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['DOUBLE', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['DATE', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['DATEV2', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['DATETIME', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['DATETIMEV2', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['DECIMALV2', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['DECIMAL32', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['DECIMAL64', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['DECIMAL128', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['VARCHAR', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['STRING', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['BITMAP', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['JSONB', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_BOOLEAN', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_TINYINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_SMALLINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_INT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_BIGINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_LARGEINT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_DATETIME', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_DATE', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_DATETIMEV2', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_DATEV2', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_FLOAT', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_DOUBLE', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_DECIMALV2', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_DECIMAL32', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_DECIMAL64', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_DECIMAL128', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_VARCHAR', '...'], 'ALWAYS_NOT_NULLABLE'],
        [['ignore'], 'BOOLEAN', ['ARRAY_STRING', '...'], 'ALWAYS_NOT_NULLABLE']
    ],

    # multi match functions
    "MultiMatch": [
        [['multi_match'], 'BOOLEAN', ['STRING', '...'], 'ALWAYS_NOT_NULLABLE']
    ]
}

# Except the following functions, other function will directly return
# null if there is null parameters.
# Functions in this set will handle null values, not just return null.
#
# This set is only used to replace 'functions with null parameters' with NullLiteral
# when applying FoldConstantsRule rules on the FE side.
# TODO(cmy): Are these functions only required to handle null values?
non_null_result_with_null_param_functions = [
    'if',
    'hll_hash',
    'concat_ws',
    'ifnull',
    'nvl',
    'nullif',
    'null_or_empty',
    'not_null_or_empty'
    'coalesce',
    'array',
    'json_array',
    'json_object'
]

# Nondeterministic functions may return different results each time they are called
nondeterministic_functions = [
    'rand',
    'now',
    'current_timestamp',
    'localtime',
    'localtimestamp',
    'curdate',
    'current_date',
    'curtime',
    'current_time',
    'utc_timestamp',
    'uuid'
]
# This is the subset of ALWAYS_NULLABLE
# The function belongs to @null_result_with_one_null_param_functions,
# as long as one parameter is null, the function must return null.
null_result_with_one_null_param_functions = [
    'str_to_date',
    'convert_tz',
    'pi',
    'e',
    'divide',
    'int_divide',
    'pmod',
    'mod',
    'fmod',
    'substr',
    'substring',
    'overlay',
    'strcmp',
    'append_trailing_char_if_absent',
    'ST_X',
    'ST_Y',
    'ST_AsText',
    'ST_GeometryFromText',
    'ST_LineFromText',
    'ST_Polygon',
    'ST_Contains',
    'from_unixtime'
]

invisible_functions = [
]
