/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */


#include "celix_launcher.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <libgen.h>
#include <signal.h>

#ifndef CELIX_NO_CURLINIT
#include <curl/curl.h>
#endif

#include "framework.h"
#include "celix_framework_factory.h"
#include "celix_constants.h"
#include "celix_framework_utils.h"

static void celixLauncher_shutdownFramework(int signal);
static void celixLauncher_ignore(int signal);
static int celixLauncher_launchWithConfigAndProps(const char *configFile, celix_framework_t* *framework, celix_properties_t* packedConfig);
static void celixLauncher_combineProperties(celix_properties_t *original, const celix_properties_t *append);
static celix_properties_t* celixLauncher_createConfig(const char* configFile, celix_properties_t* embeddedProperties);

static void celixLauncher_printUsage(char* progName);
static void celixLauncher_printProperties(celix_properties_t *embeddedProps, const char* configFile);
static void celixLauncher_printEmbeddedBundles();
static int celixLauncher_createBundleCache(celix_properties_t* embeddedProperties, const char* configFile);

#define DEFAULT_CONFIG_FILE "config.properties"

#define CELIX_LAUNCHER_OK_EXIT_CODE 0
#define CELIX_LAUNCHER_ERROR_EXIT_CODE 1

static framework_t *g_fw = NULL;



int celixLauncher_launchAndWaitForShutdown(int argc, char *argv[], celix_properties_t* packedConfig) {
    celix_framework_t* framework = NULL;
    int rc = celixLauncher_launchWithArgv(argc, argv, packedConfig, &framework);
    if (rc == 0 && framework != NULL) {
        celixLauncher_waitForShutdown(framework);
        celixLauncher_destroy(framework);
    }
    return rc;
}

int celixLauncher_launchWithArgv(int argc, char *argv[], celix_properties_t* embeddedConfig, celix_framework_t** frameworkOut) {
	celix_framework_t* framework = NULL;

	// Perform some minimal command-line option parsing...
	char *opt = NULL;
	char* configFile = NULL;
	bool showProps = false;
    bool showEmbeddedBundles = false;
    bool createCache = false;
	for (int i = 1; i < argc; ++i) {
		opt = argv[i];
		// Check whether the user wants some help...
        if (strncmp("-?", opt, sizeof("-?")) == 0 || strncmp("-h", opt, sizeof("-h")) == 0 || strncmp("--help", opt, sizeof("--help")) == 0) {
            celixLauncher_printUsage(argv[0]);
            celix_properties_destroy(embeddedConfig);
            return CELIX_LAUNCHER_OK_EXIT_CODE;
        } else if (strncmp("-p", opt, sizeof("-p")) == 0 || strncmp("--props", opt, sizeof("--props")) == 0) {
            showProps = true;
        } else if (strncmp("-c", opt, sizeof("-c")) == 0 || strncmp("--create-bundle-cache", opt, sizeof("--create-bundle-cache")) == 0) {
            createCache = true;
        } else if (strncmp("--embedded_bundles", opt, sizeof("--embedded_bundles")) == 0) {
            showEmbeddedBundles = true;
		} else {
            configFile = opt;
		}
	}

	if (configFile == NULL) {
        configFile = DEFAULT_CONFIG_FILE;
	}

	if (embeddedConfig == NULL) {
        embeddedConfig = celix_properties_create();
	}

	if (showProps) {
        celixLauncher_printProperties(embeddedConfig, configFile);
		celix_properties_destroy(embeddedConfig);
		return CELIX_LAUNCHER_OK_EXIT_CODE;
	}

    if (createCache) {
        return celixLauncher_createBundleCache(embeddedConfig, configFile);
    }

    if (showEmbeddedBundles) {
        celixLauncher_printEmbeddedBundles();
        celix_properties_destroy(embeddedConfig);
        return CELIX_LAUNCHER_OK_EXIT_CODE;
    }

	struct sigaction sigact;
	memset(&sigact, 0, sizeof(sigact));
	sigact.sa_handler = celixLauncher_shutdownFramework;
	sigaction(SIGINT,  &sigact, NULL);
	sigaction(SIGTERM, &sigact, NULL);

	memset(&sigact, 0, sizeof(sigact));
	sigact.sa_handler = celixLauncher_ignore;
	sigaction(SIGUSR1,  &sigact, NULL);
	sigaction(SIGUSR2,  &sigact, NULL);


	int rc = celixLauncher_launchWithConfigAndProps(configFile, &framework, embeddedConfig);
	if (rc == CELIX_LAUNCHER_OK_EXIT_CODE) {
		g_fw = framework;
		*frameworkOut = framework;
	}
	return rc;
}

static void celixLauncher_shutdownFramework(int signal) {
	if (g_fw != NULL) {
		celixLauncher_stop(g_fw); //NOTE main thread will destroy
	}
}

static void celixLauncher_ignore(int signal) {
	//ignoring for signal SIGUSR1, SIGUSR2. Can be used on threads
}

int celixLauncher_launch(const char *configFile, celix_framework_t* *framework) {
	return celixLauncher_launchWithConfigAndProps(configFile, framework, NULL);
}

static int celixLauncher_launchWithConfigAndProps(const char *configFile, celix_framework_t** framework, celix_properties_t* packedConfig) {
    celix_properties_t* config = celixLauncher_createConfig(configFile, packedConfig);
    return celixLauncher_launchWithProperties(config, framework);
}


int celixLauncher_launchWithProperties(celix_properties_t* config, celix_framework_t** framework) {
#ifndef CELIX_NO_CURLINIT
	// Before doing anything else, let's setup Curl
	curl_global_init(CURL_GLOBAL_ALL);
#endif
	*framework = celix_frameworkFactory_createFramework(config);
	return *framework != NULL ? CELIX_LAUNCHER_OK_EXIT_CODE : CELIX_LAUNCHER_ERROR_EXIT_CODE;
}

void celixLauncher_waitForShutdown(celix_framework_t* framework) {
    celix_framework_waitForStop(framework);
}

void celixLauncher_destroy(celix_framework_t* framework) {
    celix_frameworkFactory_destroyFramework(framework);
#ifndef CELIX_NO_CURLINIT
	// Cleanup Curl
	curl_global_cleanup();
#endif
}

void celixLauncher_stop(celix_framework_t* framework) {
    celix_framework_stopBundle(framework, CELIX_FRAMEWORK_BUNDLE_ID);
}


static void celixLauncher_printUsage(char* progName) {
    printf("Usage:\n  %s [-h|-p] [path/to/runtime/config.properties]\n", basename(progName));
    printf("Options:\n");
    printf("\t-h | --help: Show this message.\n");
    printf("\t-p | --props: Show the embedded and runtime properties for this Celix container and exit.\n");
    printf("\t-c | --create-bundle-cache: Create the bundle cache for this Celix container and exit.\n");
    printf("\t--embedded_bundles: Show the embedded bundles for this Celix container and exit.\n");
    printf("\n");
}

static void celixLauncher_printProperties(celix_properties_t *embeddedProps, const char *configFile) {
	const char *key = NULL;
	celix_properties_t *keys = celix_properties_create(); //only to store the keys

	printf("Embedded properties:\n");
	if (embeddedProps == NULL || celix_properties_size(embeddedProps) == 0) {
		printf("|- Empty!\n");
	} else {
		CELIX_PROPERTIES_FOR_EACH(embeddedProps, key) {
			const char *val = celix_properties_get(embeddedProps, key, "!Error!");
			printf("|- %s=%s\n", key, val);
            celix_properties_set(keys, key, NULL);
        }
	}
	printf("\n");

	celix_properties_t *runtimeProps = NULL;
	if (configFile != NULL) {
        runtimeProps = celix_properties_load(configFile);
	}
	printf("Runtime properties (input %s):\n", configFile);
	if (runtimeProps == NULL || celix_properties_size(runtimeProps) == 0) {
		printf("|- Empty!\n");
	} else {
		CELIX_PROPERTIES_FOR_EACH(runtimeProps, key) {
			const char *val = celix_properties_get(runtimeProps, key, "!Error!");
			printf("|- %s=%s\n", key, val);
            celix_properties_set(keys, key, NULL);
		}
	}
    printf("\n");

	//combined result
	printf("Resolved (env, runtime and embedded) properties:\n");
	if (celix_properties_size(keys) == 0) {
		printf("|- Empty!\n");
	} else {
		CELIX_PROPERTIES_FOR_EACH(keys, key) {
			const char *valEm = celix_properties_get(embeddedProps, key, NULL);
            const char *valRt = celix_properties_get(runtimeProps, key, NULL);
            const char *envVal = getenv(key);
            const char *val = envVal != NULL ? envVal : valRt != NULL ? valRt : valEm;
            const char *source = envVal != NULL ? "environment" : valRt != NULL ? "runtime" : "embedded";
			printf("|- %s=%s (source %s)\n", key, val, source);
		}
	}
    printf("\n");

	if (runtimeProps != NULL) {
		celix_properties_destroy(runtimeProps);
	}
	celix_properties_destroy(keys);
}

static void celixLauncher_printEmbeddedBundles() {
    celix_array_list_t* embeddedBundles = celix_framework_utils_listEmbeddedBundles();
    printf("Embedded bundles:\n");
    for (int i = 0; i < celix_arrayList_size(embeddedBundles); ++i) {
        char* bundle = celix_arrayList_get(embeddedBundles, i);
        printf("|- %02i: %s\n", (i+1), bundle);
        free(bundle);
    }
    if (celix_arrayList_size(embeddedBundles) == 0) {
        printf("|- no embedded bundles\n");
    }
    celix_arrayList_destroy(embeddedBundles);
}

static int celixLauncher_createBundleCache(celix_properties_t* embeddedProperties, const char* configFile) {
    celix_framework_t* fw = NULL;
    celix_properties_t* config = celixLauncher_createConfig(configFile, embeddedProperties);
    celix_status_t status = framework_create(&fw, config);
    if (status != CELIX_SUCCESS) {
        fprintf(stderr, "Failed to create framework for bundle cache creation\n");
        return CELIX_LAUNCHER_ERROR_EXIT_CODE;
    }
    status = celix_framework_utils_createBundleArchivesCache(fw);
    (void)framework_destroy(fw);
    if (status != CELIX_SUCCESS) {
        fprintf(stderr, "Failed to create bundle cache\n");
        return CELIX_LAUNCHER_ERROR_EXIT_CODE;
    }
    return CELIX_LAUNCHER_OK_EXIT_CODE;
}

static void celixLauncher_combineProperties(celix_properties_t *original, const celix_properties_t *append) {
	if (original != NULL && append != NULL) {
		const char *key = NULL;
		CELIX_PROPERTIES_FOR_EACH(append, key) {
			celix_properties_set(original, key, celix_properties_get(append, key, NULL));
		}
	}
}

static celix_properties_t* celixLauncher_createConfig(const char* configFile, celix_properties_t* embeddedProperties) {
    if (embeddedProperties == NULL) {
        embeddedProperties = celix_properties_create();
    }

    FILE *config = fopen(configFile, "r");
    if (config != NULL) {
        celix_properties_t *configProps = celix_properties_loadWithStream(config);
        fclose(config);
        if (configProps != NULL) {
            celixLauncher_combineProperties(embeddedProperties, configProps);
            celix_properties_destroy(configProps);
        }
    }

    return embeddedProperties;
}
