/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.spark.utils.streaming;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.spark.stats.BufferingInputStreamStats;
import org.apache.cassandra.spark.utils.ByteBufferUtils;
import org.apache.cassandra.spark.utils.ThrowableUtils;
import org.apache.cassandra.spark.utils.streaming.CassandraFile;
import org.apache.cassandra.spark.utils.streaming.CassandraFileSource;
import org.apache.cassandra.spark.utils.streaming.StreamBuffer;
import org.apache.cassandra.spark.utils.streaming.StreamConsumer;
import org.jetbrains.annotations.NotNull;

public class BufferingInputStream<T extends CassandraFile>
extends InputStream
implements StreamConsumer {
    private static final StreamBuffer.ByteArrayWrapper END_MARKER = StreamBuffer.wrap(ByteBufferUtils.EMPTY);
    private static final StreamBuffer.ByteArrayWrapper FINISHED_MARKER = StreamBuffer.wrap(ByteBufferUtils.EMPTY);
    private static final StreamBuffer.ByteArrayWrapper ERROR_MARKER = StreamBuffer.wrap(ByteBufferUtils.EMPTY);
    private final BlockingQueue<StreamBuffer> queue;
    private final CassandraFileSource<T> source;
    private final BufferingInputStreamStats<T> stats;
    private final long startTimeNanos;
    private volatile Throwable throwable = null;
    private volatile boolean activeRequest = false;
    private volatile boolean closed = false;
    private final AtomicLong bytesWritten = new AtomicLong(0L);
    private long rangeStart = 0L;
    private long bytesRead = 0L;
    private long timeBlockedNanos = 0L;
    private boolean skipping = false;
    private StreamState state = StreamState.Init;
    private StreamBuffer currentBuffer = null;
    private int position;
    private int length;

    public BufferingInputStream(CassandraFileSource<T> source, BufferingInputStreamStats<T> stats) {
        this.source = source;
        this.queue = new LinkedBlockingQueue<StreamBuffer>();
        this.startTimeNanos = System.nanoTime();
        this.stats = stats;
    }

    public BufferingInputStream(CassandraFileSource<T> source, BufferingInputStreamStats<T> stats, long position) {
        this(source, stats);
        this.rangeStart = position;
        this.bytesRead = position == 0L ? 0L : position + 1L;
        this.bytesWritten.set(this.bytesRead);
    }

    public BufferingInputStream<T> reBuffer(long position) {
        this.closeInternal(false);
        return new BufferingInputStream<T>(this.source, this.stats, position);
    }

    public long startTimeNanos() {
        return this.startTimeNanos;
    }

    public long timeBlockedNanos() {
        return this.timeBlockedNanos;
    }

    public long bytesWritten() {
        return this.bytesWritten.get();
    }

    public long bytesRead() {
        return this.bytesRead;
    }

    public long bytesBuffered() {
        return this.bytesWritten() - this.bytesRead();
    }

    public boolean isFinished() {
        return this.bytesWritten() >= this.source.size();
    }

    private boolean isClosed() {
        return this.state == StreamState.Closed;
    }

    public long chunkBufferSize() {
        return this.source.chunkBufferSize();
    }

    private boolean canRequestMore() {
        return !this.activeRequest && !this.skipping && !this.isBufferFull() && !this.isClosed();
    }

    private void maybeRequestMore() {
        if (this.canRequestMore()) {
            this.requestMore();
        }
    }

    private void requestMore() {
        if (this.rangeStart >= this.source.size()) {
            if (this.isFinished()) {
                this.queue.add(FINISHED_MARKER);
            }
            return;
        }
        long chunkSize = this.rangeStart == 0L ? this.source.headerChunkSize() : this.source.chunkBufferSize();
        long rangeEnd = Math.min(this.source.size(), this.rangeStart + chunkSize);
        if (rangeEnd >= this.rangeStart) {
            this.activeRequest = true;
            this.source.request(this.rangeStart, rangeEnd, this);
            this.rangeStart += chunkSize + 1L;
        } else {
            throw new IllegalStateException(String.format("Tried to request invalid range start=%d end=%d", this.rangeStart, rangeEnd));
        }
    }

    public boolean isBufferFull() {
        return this.bytesBuffered() >= this.source.maxBufferSize();
    }

    public static long timeoutLeftNanos(Duration timeout, long nowNanos, long lastActivityNanos) {
        return Math.min(timeout.toNanos(), timeout.toNanos() - (nowNanos - lastActivityNanos));
    }

    private Throwable timeoutException(Duration timeout) {
        return new TimeoutException(String.format("No activity on BufferingInputStream for %d seconds", timeout.getSeconds()));
    }

    private void timeoutError(Duration timeout) {
        this.onError(this.timeoutException(timeout));
    }

    @Override
    public void onRead(StreamBuffer buffer) {
        int length = buffer.readableBytes();
        if (length <= 0 || this.closed) {
            return;
        }
        this.bytesWritten.addAndGet(length);
        this.queue.add(buffer);
        this.stats.inputStreamBytesWritten(this.source, length);
    }

    @Override
    public void onEnd() {
        this.activeRequest = false;
        if (this.isFinished()) {
            this.queue.add(FINISHED_MARKER);
        } else {
            this.queue.add(END_MARKER);
        }
    }

    @Override
    public void onError(@NotNull Throwable throwable) {
        this.throwable = ThrowableUtils.rootCause(throwable);
        this.activeRequest = false;
        this.queue.add(ERROR_MARKER);
        this.stats.inputStreamFailure(this.source, throwable);
    }

    @Override
    public int available() {
        return Math.toIntExact(this.bytesBuffered());
    }

    @Override
    public boolean markSupported() {
        return false;
    }

    @Override
    public long skip(long count) throws IOException {
        if (count <= 0L) {
            return 0L;
        }
        if (count <= this.bytesBuffered()) {
            long actual = super.skip(count);
            this.stats.inputStreamBytesSkipped(this.source, actual, 0L);
            return actual;
        }
        this.skipping = true;
        long remaining = count;
        while ((this.activeRequest || !this.queue.isEmpty()) && (remaining -= super.skip(remaining)) > 0L) {
        }
        if (remaining > 0L) {
            this.rangeStart += remaining;
            this.bytesWritten.addAndGet(remaining);
            this.bytesRead += remaining;
        }
        this.skipping = false;
        switch (this.state) {
            case Reading: 
            case NextBuffer: {
                this.maybeRequestMore();
                break;
            }
            default: {
                this.checkState();
            }
        }
        this.stats.inputStreamBytesSkipped(this.source, count - remaining, remaining);
        return count;
    }

    public int read(ByteBuffer buffer) throws IOException {
        int read = 0;
        int remaining = buffer.remaining();
        while (read < remaining) {
            if (this.checkState() < 0) {
                throw new EOFException();
            }
            int readLength = Math.min(this.length - this.position, buffer.remaining());
            if (0 < readLength) {
                read += readLength;
                this.currentBuffer.getBytes(this.position, buffer, readLength);
                this.position += readLength;
                this.bytesRead += (long)readLength;
            }
            this.maybeReleaseBuffer();
        }
        return read;
    }

    public static void ensureOffsetWithinBounds(int offset, int length, int bufferLength) {
        if (offset < 0 || length < 0 || bufferLength < 0 || length + offset > bufferLength) {
            throw new IndexOutOfBoundsException(String.format("Out of bounds, offset=%d, length=%d, bufferLength=%d", offset, length, bufferLength));
        }
    }

    @Override
    public int read(byte[] buffer, int offset, int length) throws IOException {
        BufferingInputStream.ensureOffsetWithinBounds(offset, length, buffer.length);
        if (length == 0) {
            return 0;
        }
        if (this.checkState() < 0) {
            return -1;
        }
        int readLength = Math.min(this.length - this.position, length);
        if (readLength > 0) {
            this.currentBuffer.getBytes(this.position, buffer, offset, readLength);
            this.position += readLength;
            this.bytesRead += (long)readLength;
        }
        this.maybeReleaseBuffer();
        return readLength;
    }

    @Override
    public int read() throws IOException {
        do {
            if (this.checkState() < 0) {
                return -1;
            }
            if (this.currentBuffer.readableBytes() != 0) continue;
            this.maybeReleaseBuffer();
        } while (this.currentBuffer == null);
        int unsigned = this.currentBuffer.getByte(this.position++) & 0xFF;
        ++this.bytesRead;
        this.maybeReleaseBuffer();
        return unsigned;
    }

    @Override
    public void close() {
        this.closeInternal(true);
    }

    private void closeInternal(boolean releaseSource) {
        if (this.state == StreamState.Closed) {
            return;
        }
        if (this.state != StreamState.End) {
            this.end();
        }
        this.state = StreamState.Closed;
        this.closed = true;
        this.releaseBuffer();
        this.queue.clear();
        if (releaseSource) {
            try {
                this.source.close();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    @Override
    public void reset() throws IOException {
        throw new IOException("reset not supported");
    }

    private void maybeReleaseBuffer() {
        this.maybeRequestMore();
        if (this.position < this.length) {
            return;
        }
        this.releaseBuffer();
        this.state = StreamState.NextBuffer;
        this.stats.inputStreamByteRead(this.source, this.position, this.queue.size(), (int)((double)this.position * 100.0 / (double)this.source.size()));
    }

    private void releaseBuffer() {
        if (this.currentBuffer != null) {
            this.currentBuffer.release();
            this.currentBuffer = null;
        }
    }

    private void nextBuffer() throws IOException {
        long startNanos = System.nanoTime();
        try {
            Duration timeout = this.source.timeout();
            if (timeout != null && timeout.getSeconds() > 0L) {
                this.currentBuffer = this.queue.poll(timeout.getSeconds(), TimeUnit.SECONDS);
                if (this.currentBuffer == null) {
                    throw new IOException(this.timeoutException(timeout));
                }
            } else {
                this.currentBuffer = this.queue.take();
            }
        }
        catch (InterruptedException exception) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(exception);
        }
        long nanosBlocked = System.nanoTime() - startNanos;
        this.timeBlockedNanos += nanosBlocked;
        this.stats.inputStreamTimeBlocked(this.source, nanosBlocked);
        this.length = this.currentBuffer.readableBytes();
        this.state = StreamState.Reading;
        this.position = 0;
        if (this.currentBuffer == null) {
            throw new IOException("Obtained a null buffer from the queue");
        }
    }

    private int checkState() throws IOException {
        switch (this.state) {
            case Closed: {
                throw new IOException("Stream is closed");
            }
            case End: {
                return -1;
            }
            case Init: {
                this.requestMore();
                this.state = StreamState.NextBuffer;
            }
            case NextBuffer: {
                this.nextBuffer();
                if (this.currentBuffer == END_MARKER) {
                    return this.handleEndMarker();
                }
                if (this.currentBuffer == FINISHED_MARKER) {
                    return this.handleFinishedMarker();
                }
                if (this.currentBuffer != ERROR_MARKER) break;
                throw new IOException(this.throwable);
            }
        }
        return 0;
    }

    private int handleFinishedMarker() {
        this.releaseBuffer();
        this.end();
        this.stats.inputStreamEndBuffer(this.source);
        return -1;
    }

    private int handleEndMarker() throws IOException {
        if (this.skipping) {
            return -1;
        }
        this.releaseBuffer();
        this.maybeRequestMore();
        this.state = StreamState.NextBuffer;
        return this.checkState();
    }

    private void end() {
        this.state = StreamState.End;
        this.stats.inputStreamEnd(this.source, System.nanoTime() - this.startTimeNanos, this.timeBlockedNanos);
    }

    private static enum StreamState {
        Init,
        Reading,
        NextBuffer,
        End,
        Closed;

    }
}

